function [s, c, velp, vels]= stress_aniso2(sigma, rho, ap, bp,dp, as, bs, ds)
% [s, c, velp, vels]= stress_aniso2(sigma, rho,  ap, bp,dp, as, bs, ds)
% Computes stress-induced anisotropy using the algorithm of Mavko, Mukerji, and Godfrey
% Geophysics, 60, 1081-1087, 1995.  This version uses input coefficients describing the hydrostatic
% pressure dependence of Vp and Vs.  See aniso for version that takes vp,vs data points as inputs.
% Method takes the dry rock velocity vs. hydrostatic stress functions as inputs and transforms them 
%  to anisotropy resulting from nonhydrostatic stress.  Method assumes that the pressure dependence and
%  and anisotropy result from crack CLOSING, and not from the creation of new cracks.
% Inputs:
%    sigma 		= vector of length 3, giving the three principal effective stresses, positive in compression.
%    rho     		= rock density. must be a scalar
%    ap,bp,dp 	= coefficients in function vp = ap -bp*exp(-dp*p)
%    as,bs,ds 	= coefficients in function vs = as -bs*exp(-ds*p)

%  Outputs:  s, c   are the 6x6 compliance and stiffness tensors
%
% Written by: Gary Mavko
% Modified the pressure array: Kaushik Bandyopadhyay

% Compute isotropic stiffness functions from input velocity functions
% I generate a bunch of Vp, Vs values and then  compute the corresponding functions
p = [1:125];               				% large array of pressures
vp = ap - bp*exp(-dp*p);			% corresponding dry rock hydrostatic vp
vs = as - bs*exp(-ds*p);			% dry rock hydrostatic vs
k0    = rho.*(vp.^2 - (4/3)*vs.^2);	%dry rock bulk modulus
mu0 = rho.*vs.^2;				%dry rock shear modulus
for k = 1: length(vp),
	[siso(:, :, k),cc] = CSiso(k0(k), mu0(k)); % Create 6x6 isotropic compliance tensor vs. pressure
end;
for k = 1: length(vp),
	dsiso(:, :, k) = siso(:, :, k) - siso(:, :, length(vp));  % Subtract off high pressure end point- 6x6 soft compliance tensor
end
for k = 1: length(vp),
    w(k) =  dsiso(1,1,k) + dsiso(2,2,k) + dsiso(3,3,k) + ... 
	         2*(dsiso(1,2,k) + dsiso(1,3,k) + dsiso(2,3,k));
    a(k)  =   dsiso(1,1,k) + dsiso(2,2,k) + dsiso(3,3,k) + ... 
	         0.5*(dsiso(4,4,k) + dsiso(5,5,k) + dsiso(6,6,k));			 

     gam(k) = (a(k) - w(k))./(8*pi);            % shear-related stiffness function
     w(k) = w(k)/(2*pi);                               	% normal stiffness function
 end

% prepare to integrate stiffness functions over all crack orientations

dt = pi/100;  dp = pi/100;   		% increment of angles, for integration
theta   = [dt/2 : dt : pi/2-dt/2];             % Euler angles, giving orientation of crack normals
phi      = [dp/2 : dp : 2*pi-dp/2];
[tt,pp] = meshgrid(theta, phi);       	% grid of theta and phi covering entire range to be integrated
ss         = sin(tt);

% set up direction cosines
m(:, :, 1) = sin(tt).*cos(pp);
m(:, :, 2) = sin(tt).*sin(pp);
m(:, :, 3) = cos(tt);

% normal stress at each angle.  Simply a matrix multiply, because stress tensor is diagonal
snorm = m(:, :, 1).^2.*sigma(1) + m(:, :, 2).^2.*sigma(2) + m(:, :, 3).^2.*sigma(3);

% set up arrays of pressure function over all angles
for k = 1: size(snorm,2),
  ww(:,k) = interp1(p, w, snorm(:,k));		% evaluate w at normal stresses resolved over all angles
  gw(:,k)  = interp1(p, gam, snorm(:,k));		% evaluate gam at normal stresses resolved over all angles
end;
  
% integrate to get 6x6 compliance tensors
s           = zeros(6,6);
s(1,1) =  dt*dp*sum(sum(ww.*m(:,:,1).^4.*ss)) + ...
                 dt*dp*sum(sum(gw.* ...
	        (4*m(:,:,1).^2 - 4*m(:,:,1).^4).*ss));
s(2,2) =  dt*dp*sum(sum(ww.*m(:,:,2).^4.*ss)) + ...
                 dt*dp* sum(sum(gw.* ...
	        (4*m(:,:,2).^2 - 4*m(:,:,2).^4).*ss));
s(3,3) =  dt*dp*sum(sum(ww.*m(:,:,3).^4.*ss)) + ...
                 dt*dp* sum(sum(gw.* ...
	        (4*m(:,:,3).^2 - 4*m(:,:,3).^4).*ss));

s(1,2) =  dt*dp*sum(sum(ww.*m(:,:,1).*m(:,:,1).*m(:,:,2).*m(:,:,2).*ss)) + ...
                 dt*dp* sum(sum(gw.* ...
	        ( - 4*m(:,:,1).*m(:,:,1).*m(:,:,2).*m(:,:,2)).*ss));
s(1,3) =  dt*dp*sum(sum(ww.*m(:,:,1).*m(:,:,1).*m(:,:,3).*m(:,:,3).*ss)) + ...
                 dt*dp* sum(sum(gw.* ...
	        ( - 4*m(:,:,1).*m(:,:,1).*m(:,:,3).*m(:,:,3)).*ss));
s(2,3) =  dt*dp*sum(sum(ww.*m(:,:,2).*m(:,:,2).*m(:,:,3).*m(:,:,3).*ss)) + ...
                 dt*dp* sum(sum(gw.* ...
	        ( - 4*m(:,:,2).*m(:,:,2).*m(:,:,3).*m(:,:,3)).*ss));
s(2,1) = s(1,2);
s(3,1) = s(1,3);
s(3,2) = s(2,3);
s(4,4) =  4*dt*dp*sum(sum(ww.*m(:,:,2).*m(:,:,3).*m(:,:,2).*m(:,:,3).*ss)) + ...
                 4*dt*dp* sum(sum(gw.* ...
	        (m(:,:,3).*m(:,:,3) + m(:,:,2).*m(:,:,2) - 4*m(:,:,2).*m(:,:,3).*m(:,:,2).*m(:,:,3)).*ss));
s(5,5) =  4*dt*dp*sum(sum(ww.*m(:,:,1).*m(:,:,3).*m(:,:,1).*m(:,:,3).*ss)) + ...
                 4*dt*dp* sum(sum(gw.* ...
	        (m(:,:,1).*m(:,:,1) + m(:,:,3).*m(:,:,3) - 4*m(:,:,1).*m(:,:,3).*m(:,:,1).*m(:,:,3)).*ss));
s(6,6) =  4*dt*dp*sum(sum(ww.*m(:,:,1).*m(:,:,2).*m(:,:,1).*m(:,:,2).*ss)) + ...
                 4*dt*dp* sum(sum(gw.* ...
	        (m(:,:,1).*m(:,:,1) + m(:,:,2).*m(:,:,2) - 4*m(:,:,1).*m(:,:,2).*m(:,:,1).*m(:,:,2)).*ss));
s = s + siso(:,:,length(vp));
c = inv(s);		
velp(1) = sqrt(c(1,1)/rho);
velp(2) = sqrt(c(2,2)/rho);
velp(3) = sqrt(c(3,3)/rho);
vels(1) = sqrt(c(4,4)/rho);
vels(2) = sqrt(c(5,5)/rho);
vels(3) = sqrt(c(6,6)/rho);
