function [wz,pz,tf]=kennettQ2(lyr,wvlt,dt,mopt,fs,disp,Q,omega0);
% KENNET CAUSAL algorithm with Q for Synthetic seismograms for plane wave, normal incidence propagation
% in 1-D layered media. This algorithm uses a nearly constant Q model proposed by Kennett, with
% a velocity correction given by Aki and Richards.
% When Q becomes very large, the results should converge with the usual elastic Kennett algorithm.
%
%[WZ,PZ,TF]=KENNETQ(LYR,WVLT,DT,MOPT,FS,DISP,Q,omega0)
%
%	WZ:    seismogram at the top (particle displacement)
%	PZ:    seismogram at bottom of layers (particle displacement)
%	TF:    [frequency, reflectivity, transmissivity]
%	       Transfer function (complex) of the layered medium
%	LYR:   [velocity, density, thickness] of layers
%          velocities should correspond to frequency equal to infinite
%	       LYR is a matrix of 3 columns with 
%             number of rows = number of layers
%	WVLT:  input source wavelet vector
%	       use [] to specify default wavelet (see also SOURCEWVLT)
%	DT:    time sampling interval of wavelet
%   MOPT: =0 for primaries only, =1 for 1st order multiples
%             anything else calculates all reverberations 
%	FS:    option for free surface reverberations.
%             FS=0 for no free surface multiples, FS=1 for free surface effect
%	DISP:  n (n < number of layers) (optional parameter)
%	       display option to show the reflectivity and transmissivity
%	       as it is being calculated recursively from the bottom of
%	       the layer stack to the top. Not shown when DISP is unspecified
%	       = -1 prevents display of any seismogram.	      
%  Q:     Quality factor (the inverse of the attenuation): it is a measure
%         of how dissipative the material is. Typical values are: 50,100, 200...
% OMEGA0: Reference frequency for Q
%
% Without any output arguments, KENNET plots the transmitted and
% reflected seismograms

% Adapted from Mukerji's Kennet, by Manuel Diaz to add finite Q
% May 17, 2001
%
% Increased vectorization for faster computation by Gary Mavko, Nov 2001
% Also increased the number of comments.

i=sqrt(-1);
if (length(wvlt)==0)  
   wvlt=sourcewvlt;                % defaults to Claerbout's wavelet
end; 
wvlt=[wvlt(:)]';                   % ensure wavelet is a row vector

[nlr,nc] = size(lyr);              % dimensions of input Earth model
ro = lyr(:,2);                     % column of layer densities
v  = lyr(:,1);                     % column of layer velocities
[v1,v2] = size(v);
d = lyr(:,3);                      % column of layer thicknesses
n=length(wvlt); 
om=(2*pi/(n*dt))*[0:1:(n/2 -1)];   % ROW vector of frequencies
[om1,om2] = size(om);

v  = repmat(v,1,om2);              % repeat velocity column by the number of frequencies
om = repmat(om,v1,1);              % repeat frequency row by the number of layers
om(:,1)=0.000001;
ro = repmat(ro,1,om2);             % repeat density column by the number of frequencies

if length(Q)==1
    Q=Q.*ones(v1,1);               % make column of Q, in case only a scalar is input
end
Q=repmat(Q,1,om2);                 % repeat Q column by the number of frequencies

% Kennett nearly constant Q dispersion model
Qw=Q.*(ones(v1,om2)-((pi.*Q).^(-1)).*log(0.577.*om/omega0));
Qwliu=Qw;

% Modified velocity, from Aki and Richards
v=v.*(ones(v1,om2) +log(om/omega0)./(pi*Qw) -i./(2*Qw));


freq  = om(1,:)/(2*pi);
p0    = ifft(wvlt); 

rdhat = zeros(size(om(1,:))); 
tdhat = ones(size(om(1,:)));

deno = ones(size(v)); rd = ones(size(v)); td = ones(size(v));
deno(2:nlr,:) = ro(2:nlr,:).*(v(2:nlr,:)) + ro(1:nlr-1,:).*(v(1:nlr-1,:));
rd(2:nlr,:)   = (ro(1:nlr-1,:).*(v(1:nlr-1,:)) - ro(2:nlr,:).*(v(2:nlr,:)))./deno(2:nlr,:);
td(2:nlr,:)   = 2*sqrt(ro(2:nlr,:).*(v(2:nlr,:)).*ro(1:nlr-1,:).*(v(1:nlr-1,:)))./deno(2:nlr,:);

if fs==1, 
      rd(1,:) = -1*rd(1,:); 
      td(1,:) =  1*td(1,:); 
else 
      rd(1,:) = 0*rd(1,:); 
      td(1,:) = 1*td(1,:); 
end
   
ru=-rd; tu=td;

for j=nlr:-1:1
    ed=exp(i.*(d(j)./v(j,:)).*om(j,:));
    if mopt==0
	    reverb=ones(size(om(j,:)));
    elseif mopt==1
        reverb=ones(size(om(j,:)))+ru(j,:).*ed.*rdhat.*ed;
    else
	    reverb=1./(1-ru(j,:).*ed.*rdhat.*ed);
    end
    rdhat=rd(j,:)+tu(j,:).*ed.*rdhat.*ed.*reverb.*td(j,:);
	tdhat=tdhat.*ed.*reverb.*td(j,:);
    if mopt==1
       dx=find(real(rdhat)>1); rdhat(dx)=ones(size(dx))+i*imag(rdhat(dx));
       dx=find(imag(rdhat)>1); rdhat(dx)=real(rdhat(dx))+i*ones(size(dx));
       dx=find(real(tdhat)>1); tdhat(dx)=ones(size(dx))+i*imag(tdhat(dx));
       dx=find(imag(tdhat)>1); tdhat(dx)=real(tdhat(dx))+i*ones(size(dx));
    end
    if   (nargin>5)  
       if disp~=-1
           if (rem(j,disp)==0)  
               figure(1),plot(freq, abs(tdhat));
               title(['Transmissivity: layer ',num2str(j)]); 
               xlabel('freq. Hz'), ylabel('|T(f)|'); drawnow
               figure(2),plot(freq, abs(rdhat));
               title(['Reflectivity: layer ',num2str(j)]); 
               xlabel('freq. Hz'), ylabel('|R(f)|'); drawnow
           end;  
	   end;
	end;
end;

tf=[freq(:), rdhat(:), tdhat(:)];
pz=tdhat.*p0(1:length(om(1,:)));
wz=rdhat.*p0(1:length(om(1,:)));
    
fltr=hanning(n/2); fltr=[ones(n/4,1);fltr(n/4+1:n/2)]'; 
pz=pz.*fltr; wz=wz.*fltr;
pz=[pz(1),pz(2:length(pz)),0,conj(fliplr(pz(2:length(pz))))];
wz=[0,wz(2:length(wz)),0,conj(fliplr(wz(2:length(wz))))];
  
pz=real(fft(pz)); wz=real(fft(wz));


if ( (nargin>5) & (disp~=-1) )
    figure(1),plot([0:dt:dt*(length(pz)-1)],(pz)),grid on,title('transmitted seismogram');
    figure(2),plot([0:dt:dt*(length(wz)-1)],wz),grid on,title('reflected seismogram');
end;

%dont bother plotting
return;

figure
plot([0:dt:dt*(length(wz)-1)],wz,'-m');
hold on;
[wz2,pz,tf]=kennett_aux(lyr,wvlt,dt,mopt,fs,disp);
plot([0:dt:dt*(length(wz)-1)],wz2,'-g');
title('Synthetic Seismograms');
legend('with attenuation Aki-Richards','without attenuation');
grid on;

figure;
plot(om(1,:),abs(1./Qwliu(1,:)),'-k');
title('1/Q vs. W (Aki-Richards model)');
xlabel('W');ylabel('1/Q');
set(gca,'XScale','log');
grid on;

figure;
plot(om(1,:),real(v(1,:)),'-k');
set(gca,'XScale','log');
title('P-velocity vs. W for the Aki-Richards Model');
xlabel('W'); ylabel('P-Velocity');
grid on;

