function [c,rho]=bkusrunavg(z,vp,vs,den,win)
%        [c,rho]=bkusrunavg(z,vp,vs,den,win)
%
% Running Backus average of log curves to yield an array of effective anisotropic
% stiffness tensors, c(6,6,N), where N is the number of depth points. At a
% particular depth point, j, the stiffness tensor is c(6,6,j).  
% Note that the output stiffness array is 3-dimensional.  If an application
% requires a 2-dimensional array at a single depth point, then use Matlab
% function SQUEEZE, e.g.  
%          squeeze(c(6,6,j))
% yields a 6x6 array.
% The Backus average is computed in a running window, and it is assumed that the
% medium consists of thin isotropic layers.
%
% input:   z       - Vector of depths, as in a well log. Can be in any units.  
%                    Depth points should be equally spaced.
%          vp      - Vector of P velocity, as in a well log -- same length
%                    as vector of depths.
%          vs      - Vector of S velocity -- same length
%                    as vector of depths.
%          den     - Vector of densities, as in a well log -- same length
%                    as vector of depths.
%          win     - length of window for averaging - in the same length units as z
% output:  c       - 6x6xN stiffness array expressed in Voigt notation -- as 
%                    defined in The Rock Physics Handbook
%          rho     - Average density at each depth point, averaged over the
%                    same window as the elastic constants
%
% see also BKUS, BKUSLOG, BKUSC

% Written by Gary Mavko 10/02
%

nwin = floor(win/mean(diff(z)));

% Create layer thicknesses from log depths
dz = diff(z); dz(end+1) = dz(end);

%
% compute isotropic lambda and mu for each thin layer
%
mu  = den.*vs.*vs;
lam = den.*vp.*vp - 2*mu;
%
% perform running Backus averages
%
x = movmean(mu.*(lam+mu)./(lam+2*mu),nwin);
y = movmean(mu.*lam./(lam+2*mu),nwin);
z = movmean(lam./(lam+2*mu),nwin);
u = movmean(1./(lam+2*mu),nwin);
v = movmean(1./mu,nwin);
w = movmean(mu,nwin);
%
% assign to 6x6 format
%
A = 4*x + z.*z./u;
B = 2*y + z.*z./u;
E = 1./u;
F = z./u;
D = 1./v;
M = w;
%
    
c(:,:,:) = zeros(6,6,length(z));
c(1,1,:) = A;
c(1,2,:) = B;
c(1,3,:) = F;
c(2,1,:) = B;
c(2,2,:) = A;
c(2,3,:) = F;
c(3,1,:) = F;
c(3,2,:) = F;
c(3,3,:) = E;
c(4,4,:) = D;
c(5,5,:) = D;
c(6,6,:) = M;

%
rho=movmean(den,nwin);
