function [g,gR,gI,Qinv,fin] = viscoelastic(f,g0,ginf,visc,alpha,gamma)
% g = viscoelastic(f,g0,ginf,visc,alpha,gamma)
% Computes viscoelastic modulus, including Standard Linear Solid and its limiting
% cases (Maxwell, Voigt, Newtonian Fluid, Cole-Cole, Havriliak-Tagami)
%
% Inputs:
%    f      frequency (can be a vector or scalar)
%    g0     zero-frequency modulus
%           =0 for Maxwell Solid or Newtonian Fluid
%    ginf   very high frequency modulus
%           = [] (with g0=0) to simulate Newtonian fluid
%           = [] (with g0>0) to simulate Voigt fluid
%    visc   viscosity (can be a vector or scalar)
%    alpha  (optional) Cole-Cole coefficient (Only use if g0=0); default: alpha=0;
%    gamma  (optional) Havriliak-Negami coefficient (Only use if g0=0); default: gamma=1;
% Outputs:
%    g      complex modulus.  May be a scalar, vector, or array g(i,j), depending
%           on input dimensions.  First dimension spans over ith frequencies for
%           each jth viscosity.
%    gR     real part of modulus
%    gI     imaginary part of modulus
%    Qinv   1/Q
%    fin    frequencies where properties are determined
%
% For general Standard Linear Solid, similar to the notation in
% Mavko et al, Rock Physics Handbook.
%
% Special cases:
%    If g0=0, reduces to Maxwell Material
%    If g0>0, and ginf=[]; represents ginf-->infinity then computes Voigt Material
%    if g0=0, and ginf=[], then simulates Newtonian fluid

% Example to compute visocity over a range of temperatures and then compute
% an array of moduli over variable f and T
%   f=10.^[0:.1:20]; T=[0:10:100]; 
%   visc = viscosityBeggsRobinson(T,5); 
%   g = viscoelastic(f,1e9,10e9,visc);
%   figure; plot(abs(g()))

fin = f;
[f,visc]= meshgrid(visc,f);

% Cole-Coefficient
if nargin<5, alpha = 0; end;

% Havriliak-Negami Coefficient
if nargin<6, gamma = 1; end;

if alpha~=0 & g0~=0, warndlg(['specified alpha=',num2str(alpha,4),' not consistent with g0=',num2str(g0,4)]); return; end;
if gamma~=1 & g0~=0, warndlg(['specified gamma=',num2str(gamma,4),' not consistent with g0=',num2str(g0,4)]); return; end;

w = 2*pi.*f;


% compute complex modulus
if isempty(ginf),                           % Newtonian Fluid or Voigt Material
    g = g0 + i*w.*visc; 
%    g = g0 + i*2*w.*visc; 
elseif g0==0 & (alpha~=0 | gamma~=1),       % Cole-Cole or Havriliak-Nagami
    g = ginf.*(1-1./(1+(i*w.*visc./ginf).^(1-alpha)).^gamma);

else,                                       % Maxwell or Standard Linear Solid
    if ginf==g0,
        g = g0*ones(size(f));
    else,
        e = g0.*ginf./(ginf-g0);
        g = ginf.*(e + i*w.*visc)./(ginf + e + i*w.*visc);
    end;
end;

gR = real(g);
gI = imag(g);
Qinv = gI./gR;
