function [s, snmo, ttop, x, ssxx] = rtppaniso(z,vp,vs,rho,e1,e2,d1,d2,gam,plane,deltax,nx,wvlt,deltat,nshift);
% [s, snmo, ttop, x, ssxx] = rtppaniso(z,vp,vs,rho,e1,e2,d1,d2,gam,plane,deltax,nx,wvlt,deltat,nshift);
%
% Computes P-wave cdp gather via raytracing in a symmetry plane of a 1-D orthorhombic Earth model.  
% Travel times, angles and offsets are computed via raytracing.  Amplitudes at each interface
% are computed using Ruger's expressions.  Final reflectivity series is convolved with
% a wavelet.  Small anisotropy is assumed, so that Thomsen parameters apply.
%
% The input model is converted to equal sample intervals in time, equal to the time 
% interval of the input wavelet.  This obviously resamples model. The ray tracing 
% times, horizontal distance of propagation, and ray angles are computed on the original 
% model, and only the reflectivities are computed after resampling to time.
% Resampling is done carefully -- if layers are thin, then there is first oversampling
% at least once per layer, then velocities are averaged using Backus, and density 
% and angles are averaged linearly back to the wavelet sample interval.
%
% Ray geometries and times are computed on a broad fan of rays shot from the surface.  
% Reflectivities are computed at the desired offsets by interpolating angles and times 
% from the fan.  Reflectivities and transmission coefficients are computed using Zoeppritz 
% at the resampled layers, equally spaced in time.  
% Two reflectivity sequences are computed:  one with moveout and the other without.
% Then convolution creates trace.  Hence, there is no NMO stretch.
% 
% Calculation is for precritical only -- rays are prevented from critical
% *****  we recommend that the wavelet be sampled at least 15 samples per wavelength, 
%        in order to ensure that the depth-to-time converted Earth model is sufficiently
%        well sampled.
%
% Inputs
%    z			Depths to BOTTOM of layers.  First z is thickness of first layer.
%               Can also be thought of as depth to reflectors.  First z is depth 
%               to first reflector. Computation starts from first depth point.
%	 vp      	log of vertical P-wave velocity
%    vs         sqrt(c44/rho); shear polarized in y-direction
%    e1,e2,
%    d1,d2, gam Generalized Thomsen parameters as defined by Tsvankin
%    plane      = 13 if propagation is in x1-x3 plane;  = 23 if propagation is in x2-x3 plane; 
%    deltax     offset interval of desired traces.  First trace is normal incidence
%    nx         number of traces to compute
%    wvlt		input wavelet
%    deltat     sample interval of input wavelet and output seismogram (in seconds)
%    nshift     number of samples to shift, if wavelet is two-sided.  normally floor(length(wvlt)/2)
%
% Outputs
%    s          array of seismograms, one per column, moveout corrected
%    snmo       array of seismograms, one per column, with moveout
%    ttop       2-way time to top of model
%    x          vector of computed offsets

% written by Gary Mavko, October, 2001

% ensure that log inputs are column vectors
z   = z(:);
vp  = vp(:);
vs  = vs(:);
rho = rho(:);
d1  = d1(:);
d2  = d2(:);
e1  = e1(:);
e2  = e2(:);
gam = gam(:);
if length(e1) == 1, e1 = e1*ones(size(vp)); end;
if length(d1) == 1, d1 = d1*ones(size(vp)); end;
if length(e2) == 1, e2 = e2*ones(size(vp)); end;
if length(d2) == 1, d2 = d2*ones(size(vp)); end;
if length(gam) == 1, gam = gam*ones(size(vp)); end;

% get rid of nonunique depth points
[zzz,j]=unique(z);
z1 = z(1); z = z(j); z(1)=z1;
vp1 = vp(1); vp = vp(j); vp(1) = vp1;    % restores first velocity, incase nonuniqueness causes us to lose it
vs1 = vs(1); vs = vs(j); vs(1) = vs1;
rho1 = rho(1); rho = rho(j); rho(1) = rho1;
d11 = d1(1); d1  = d1(j); d1(1) = d11;
d21 = d2(1); d2  = d2(j); d2(1) = d21;
e11 = e1(1); e1  = e1(j); e1(1) = e11;
e21 = e2(1); e2  = e2(j); e2(1) = e21;
gam1 = gam(1); gam = gam(j); gam(1) = gam1;

% select parameters for the desired symmetry plane
if plane==13,
	e = e2; d = d2;
elseif plane==23,
	e = e1; d = d1;
end;

% compute interval thicknesses.  First layer extends from surface to first log depth point.
h(1) = z(1); h(2: length(z)) = diff(z); h = h(:);

% set up array of desired offsets for computed traces
x = [0: deltax: (nx-1)*deltax]; 

% Form vector of starting ray angles, starting from zero
% This  fan of rays will be interpolated after ray tracing to get desired ray offsets.
nthetas = 35;                                               % number of rays to shoot
thetamax = 40;                                              % angle in degrees of maximum ray to shoot
sinlim = 0.98;                     % maximum allowed sin(theta) during ray tracing (avoid critical)
%sinthetas = sin((pi/180)*linspace(0, 40, nthetas));        % layer 1 phase angles, each ray
sinthetas = linspace(0, sin((pi/180)*thetamax), nthetas);	% fan of rays, equal in sin(theta)
p = sinthetas./(vp(1).*(1+d(1).*sinthetas.^2 + (e(1)-d(1)).*sinthetas.^4));	 % ray parameters, layer 1

ss   = repmat(sinthetas, length(vp), 1);
pp   = repmat(p, length(vp), 1);
dd   = repmat(d,  1, nthetas);
ee   = repmat(e,  1, nthetas);
vv   = repmat(vp, 1, nthetas);
hh   = repmat(h,  1, nthetas);

%  Compute Snell's law for each ray and each layer to get phase and ray angles.  The anisotropy makes
%  Snells very nonlinear, because phase velocity depends on the unknown angle.  Therefore, we iterate.

% first guess of phase velocity in all layers -- based on straight rays, same phase angle all layers
vphase = vv.*(1+dd.*ss.^2 + (ee-dd).*ss.^4); 

% now iterate
for k = 1:10,
	ss     = vphase.*pp;                             % next guess at phase angles, based on Snell's law
	ss(ss<0) = 0; ss(ss>.9) = .9;
    vphase = vv.*(1+dd.*ss.^2 + (ee-dd).*ss.^4);     % phase velocity
end;
cc     = sqrt(1 - ss.^2);                            % cosine of phase angle
dvdth  = vv.*(2.*dd.*ss.*cc + 4*(ee-dd).*ss.^3.*cc); % derivative of phase velocity
phiray = asin(ss) + atan(dvdth./vphase);             % ray angle
tt     = tan(phiray);                                % tan of ray angle
vray   = sqrt(vphase.^2 + dvdth.^2);                 % ray velocity

%  Compute array of dx - increments of traveled distance in horizontal direction
%  within each layer, for each ray
dx = hh.*tt;  % one-way horizontal propagation distance

%  Compute arry of dt - increment of travel time within each layer, for each ray
dt = sqrt(hh.^2 + dx.^2)./vray;  % one-way time

%  Sum to get total travel distance and two-way time along each ray
xcum = 2.*cumsum(dx);
tcum = 2.*cumsum(dt);

% Compute vertical travel time
t0    = 2*cumsum(h./vp);              % Normal incidence time, unequally spaced.
ttop  = t0(1);                        % Time to top of model
tbot  = t0(end);                      % Time to bottom of model

% Convert layer properties from depth to equally spaced in time.  Logs are now time converted.
% Logs are first oversampled for interpolation, then blocked. 
dtavg = 1.0*mean(h(2:end)./vp(2:end));               %half average travel time through the model layers
nover = ceil(deltat/dtavg);            % oversample factor to take wavelet sample to dtavg
nover = max(1,nover) ;                 % oversample factor
time  = [ttop: deltat/nover: tbot]';   % Vector of equally sampled time, finely spaced

% convert all layer quantities to equally spaced in time.
% vp, vs, and rho are interpolated with special function that references 'next' increment, 
% since input quantities are referenced to layer bottoms.
vpt    = interpnext(t0, vp,   time);  
vst    = interpnext(t0, vs,   time);  
rhot   = interpnext(t0, rho,  time); 
d1t    = interpnext(t0, d1,   time); 
d2t    = interpnext(t0, d2,   time); 
e1t    = interpnext(t0, e1,   time); 
e2t    = interpnext(t0, e2,  time); 
gamt   = interpnext(t0, gam,  time); 
sst    = interp1(t0,    ss,   time);  
tcumt  = interp1(t0,    tcum, time);  
xcumt  = interp1(t0,    xcum, time);
if nover > 1  , 
	vpt   = blockav(1./vpt, nover);  vpt = 1./vpt(1:nover:end);
	vst   = blockav(1./vst, nover);  vst = 1./vst(1:nover:end);
	rhot  = blockav(rhot, nover);   rhot = rhot(1:nover:end);
	d1t   = blockav(d1t, nover);     d1t = d1t(1:nover:end);
	d2t   = blockav(d2t, nover);     d2t = d2t(1:nover:end);
	e1t   = blockav(e1t, nover);     e1t = e1t(1:nover:end);
	e2t   = blockav(e2t, nover);     e2t = e2t(1:nover:end);
	gamt  = blockav(gamt, nover);   gamt = gamt(1:nover:end);
	sst   = blockav(sst, nover);     sst = sst(1:nover:end, :);
	tcumt = tcumt(1:nover:end, :);                              
	xcumt = xcumt(1:nover:end, :);                          
end;

% quantities are now in layers equally sampled in vertical time.  
% First time sample tcumt is ray time to top of model
% First entry in xcumt is offset to top of model

% now compute Ruger reflectivity at each layer boundary - equally spaced in time
rflag = diff(vpt)==0 & diff(vst)==0 & diff(rhot)==0;   % flag for layers with no contrast
lenrppt = 2*length(vpt);
Rppt = zeros(lenrppt, nx);                             % initialize equally spaced Rpp to zero; first row at top of model
%nlength = 20 + length(vpt);
%Rppt = zeros(nlength, nx);                             % initialize equally spaced Rpp to zero; first row at top of model
Rpptnmo = Rppt;                                        % moved out reflectivities
ssxx    = nan*Rppt;                                    % angles
for i = 1:length(vpt)-1,                               % loop over layers
	if ~rflag(i),                                      % only compute at boundaries with contrast
	    ttx = interp1(xcumt(i+1,:), tcumt(i+1,:), x);  % 2-way time to this reflector, each desired offset
	    ssx = interp1(xcumt(i,:), sst(i,:), x);        % sintheta at this reflector, each offset
        ssxx(i,:) = ssx;
		if plane == 13
	       R         = rugersym13(vpt(i),  vst(i),   rhot(i),   d2t(i),   e2t(i),   gamt(i), ...
	                              vpt(i+1), vst(i+1), rhot(i+1), d2t(i+1), e2t(i+1), gamt(i+1), ...
					              (180/pi)*asin(ssx));
		elseif plane == 23
	       R         = rugersym23(vpt(i),  vst(i),   rhot(i),   d1t(i),   e1t(i), ...
	                              vpt(i+1), vst(i+1), rhot(i+1), d1t(i+1), e1t(i+1), ...
					              (180/pi)*asin(ssx));
		end;
        R(ssx >= sinlim-.02) = 0.;                      % mute at maximum allowed sin(theta)
        R(~isreal(R)) = 0.; R=real(R);                  % mute if reflectivity is becoming complex
        Rppt(i+1,:)=R;
%       find integer time step closest to moveout time ttx
%       nmofac gives weight, maximum when ttx falls right on integer time step
		nmo = max(1,min(1+floor((ttx-ttop)/deltat), lenrppt)); 
		nmofac = abs(nmo-(ttx-ttop)/deltat);
		nmonext = max(1,min(nmo+1,lenrppt));
		for j=1:nx,     % moved out reflectivity is divided between two nearest time steps
			Rpptnmo(nmo(j),     j) = Rpptnmo(nmo(j),     j) + Rppt(i+1,j)*nmofac(j);
			Rpptnmo(nmonext(j), j) = Rpptnmo(nmonext(j), j) + Rppt(i+1,j)*(1-nmofac(j));
		end; 
	end;
end;
Rppt    = Rppt(1:40+length(vpt),:);
size(Rppt)
Rpptnmo = Rpptnmo(1:40+length(vpt),:);
Rppt(isnan(Rppt)) = 0; Rpptnmo(isnan(Rpptnmo))=0;
Rppt(end, :) = 0;      Rpptnmo(end, :) = 0;
ssxx    = ssxx(1:40+length(vpt),:);
for k = 1:nx, ssxx(:,k)    = fillnan(ssxx(:,k)); end;

% convolve with wavelet
for k = 1:nx
	s(:,k)    = conv(Rppt(:,k), wvlt);
	snmo(:,k) = conv(Rpptnmo(:,k), wvlt);
end;
s(1:nshift, :) = [];         % apply shift, if wavelet is non-causal
snmo(1:nshift, :) = [];      % apply shift, if wavelet is non-causal

if nargout == 0, 
  figure;seisplot(s,ttop,deltat,0, deltax)
  figure;seisplot(snmo,ttop,deltat,0, deltax)
end;

%---------------------------------------------------------------------
function [logb]=blockav(log, nb)
%logb = blockav(log,nb)
%
%Block average of input LOG over NB points. LOG can be a single column
%vector or multiple column matrix with Nan as missing values. 
%LOG is padded by the last row if the total number of rows in LOG is 
%not an integer multiple of NB. Output is the blocked log LOGB with
%the same number of rows as LOG.

% Written by T. Mukerji, 1998

if size(log,1)==1,log=log(:); end; [nr,nc]=size(log);

npad=nb-rem(nr,nb);
if npad~=0, lrow=log(end,:); logpad=[log;lrow(ones(npad,1),:)]; end;

logpad=reshape(logpad,nb,(nr+npad)/nb,nc); logpad=nanmean(logpad);
logpad=reshape(logpad(ones(nb,1),:,:),nr+npad,nc);
logb=logpad(1:nr,:);

%---------------------------------------------------------------------
function  y=interpnext(x1, y1, x)
% y = interpnext(x1, y1, x)
%
% interpolation, similar to interp1, except that it uses the 'next' option.
% For desired new sample at x, the input function is sampled at the next sample x1
% greater or equal to x
% Method alters input data, by turning each point into a pair, so that normal
% interpolation is fooled.

dx = max(100*eps,mean(diff(x1))*1e-06);

x1(end) = max(x1(end),max(x) + dx);
y1(end+1) = y1(end);
n1 = length(x1);
n  = length(x);

xx1(1:2:2*n1-1) = x1;
xx1(2:2:2*n1)   = x1+dx;
yy1(1:2:2*n1-1) = y1(1:end-1);
yy1(2:2:2*n1)   = y1(2:end);

y = interp1(xx1,yy1,x);

%---------------------------------------------------------------------
function Rpp = rugersym13(vpa,vsa,rhoa,d2a,e2a,gammaa,vpb,vsb,rhob,d2b,e2b,gammab,phi);
% Rpp = rugersym13(vpa,vsa,rhoa,d2a,e2a,gammaa,vpb,vsb,rhob,d2b,e2b,gammab,phi);
% computes pp reflectivity in the x1-x3 symmetry plane of orthorhombic medium, using algorithm of
% Ruger, Geophysics, May-June 1998
%
% Inputs corresponding to layer (a) and layer (b)
%   Vpa,Vpb	 	vertical P-wave velocity
%   Vsb,Vsb		vertical S-wave velocity, polarized in the X2 direction = sqrt(c44/rho)
%   rhoa,rhob	densities
%	d2a,d2b     modified Thomsen parameter d2 = [(c13+c55)^2-(c33-c55)^2]/[2c33*(c33-c55)]
%   e2a,e2b     modified Thomsen parameter e2 = (c11-c33)/2c33
%   gammaa,b    modified Thomsen parameter gamma = (c44-c55)/2c55
%   phi         angle of incidence in degrees

dz  = (rhob.*vpb - rhoa.*vpa)./(rhob.*vpb + rhoa.*vpa);
dvp = (vpb - vpa)./(vpb + vpa);
dg  = (rhob.*vsb.^2 - rhoa.*vsa.^2)./(rhob.*vsb.^2 + rhoa.*vsa.^2);

Rpp = dz + ...
      (dvp-4.*(((vsb+vsa)./(vpb+vpa)).^2).*(dg -(gammab-gammaa)) +0.5*(d2b-d2a)).*(sin((pi/180)*phi)).^2 + ...
	  (dvp + 0.5*(e2b - e2a)).*(sin((pi/180)*phi).^2).*tan((pi/180)*phi).^2;
	  
%---------------------------------------------------------------------
function Rpp = rugersym23(vpa,vsa,rhoa,d1a,e1a,vpb,vsb,rhob,d1b,e1b,phi);
% Rpp = rugersym23(vpa,vsa,rhoa,d1a,e1a,gammaa,vpb,vsb,rhob,d1b,e1b,phi);
% computes pp reflectivity in the x2-x3 symmetry plane of orthorhombic medium, using algorithm of
% Ruger, Geophysics, May-June 1998
%
% Inputs corresponding to layer (a) and layer (b)
%   Vpa,Vpb	 	vertical P-wave velocity
%   Vsb,Vsb		vertical S-wave velocity, polarized in the X2 direction = sqrt(c44/rho)
%   rhoa,rhob	densities
%	d1a,d1b     modified Thomsen parameter d1 = [(c23+c44)^2-(c33-c44)^2]/[2c33*(c33-c44)]
%   e1a,e1b     modified Thomsen parameter e1 = (c22-c33)/2c33
%   phi         angle of incidence in degrees

dz  = (rhob.*vpb - rhoa.*vpa)./(rhob.*vpb + rhoa.*vpa);
dvp = (vpb - vpa)./(vpb + vpa);
dg  = (rhob.*vsb.^2 - rhoa.*vsa.^2)./(rhob.*vsb.^2 + rhoa.*vsa.^2);

Rpp = dz + ...
      (dvp-4.*(((vsb+vsa)./(vpb+vpa)).^2).*dg +0.5*(d1b-d1a)).*(sin((pi/180)*phi)).^2 + ...
	  (dvp + 0.5*(e1b - e1a)).*(sin((pi/180)*phi).^2).*tan((pi/180)*phi).^2;

	  
	  
	  
	  
