function [s, snmo, ttop, x, ssxx] = rtcdppsQ(z, vp, vs, rho, deltax, nx, wvlt, deltat, nshift, Qp, Qs);
%        [s, snmo, ttop, x, ssxx] = rtcdppsQ(z, vp, vs, rho, deltax, nx, wvlt, deltat, nshift, Qp, Qs);
%
% Computes P-S cdp gather via raytracing in a 1-D Earth model.  Travel times,
% angles and offsets are computed via raytracing.  Amplitudes at each interface
% are computed using Zoeppritz.  Final reflectivity series is convolved with
% a wavelet.  Attenuation is estimated using a simple, causal, constant Q
% model along the ray paths, without additional effects of focusing.  This
% is implemented by placing a short minimum phase filter at each reflector
% in the time domain
% Calculation is for precritical only -- rays are prevented from critical
% *****  we recommend that the wavelet be sampled at least 15 samples per wavelength, 
%        in order to ensure that the depth-to-time converted Earth model is sufficiently
%        well sampled.
%
% Inputs
%    z			Depths to BOTTOM of layers.  First z is thickness of first layer.
%               Can also be thought of as depth to reflectors.  First z is depth 
%               to first reflector. Computation starts from first depth point.
%               Can think of z as the depth column in a well log.
%	 vp,vs,rho	Interval properties corresponding to the depths
%    deltax     offset interval of desired traces.  First trace is normal incidence
%    nx         number of traces to compute
%    wvlt		input wavelet
%    deltat     sample interval of input wavelet and output seismogram (in seconds)
%    nshift     number of samples to shift, if wavelet is two-sided.  normally floor(length(wvlt)/2)
%    Qp,Qs      Q.   Can be vectors of values for each model layer, or scalars
%
% Outputs
%    s          array of seismograms, one per column, moveout corrected
%    snmo       array of seismograms, one per column, with moveout
%    ttop       2-way time to top of model
%    xout       computed offsets, should be approximately equal to the input x
%    ssxx       array of sin(theta), giving angles of incidence
%               corresponding to each seismogram


% written by Gary Mavko, October, 2001

% ensure that log inputs are column vectors
z   = z(:);
vp  = vp(:);
vs  = vs(:);
rho = rho(:);
Qp   = Qp(:);
if length(Qp)==1, Qp = Qp*ones(size(vp)); end;
Qs   = Qs(:);
if length(Qs)==1, Qs = Qs*ones(size(vp)); end;

% get rid of nonunique depth points
[zzz,j]=unique(z);
z1 = z(1); z = z(j); z(1)=z1;
vp1 = vp(1); vp = vp(j); vp(1) = vp1;    % restores first velocity, incase nonuniqueness causes us to lose it
vs1 = vs(1); vs = vs(j); vs(1) = vs1;
rho1 = rho(1); rho = rho(j); rho(1) = rho1;
Qp1 = Qp(1); Qp  = Qp(j); Qp(1) = Qp1;
Qs1 = Qs(1); Qs  = Qs(j); Qs(1) = Qs1;

% compute interval thicknesses.  First layer extends from surface to first log depth point.
h(1) = z(1); h(2: length(z)) = diff(z); h = h(:);

% set up array of desired offsets for computed traces
x = [0: deltax: (nx-1)*deltax]; 

% Form an initial vector of starting ray angles, ranging from zero, to a max
% corresponding to a little past that corresponding to a straight ray.
% This initial fan of rays will be interpolated after ray tracing to get desired ray offsets.
nthetas = 25;                      % initial number of ray fans to shoot
thetamax = 50;                     % angle in degrees of maximum ray to shoot
sinlim = 0.98;                     % maximum allowed sin(theta) during ray tracing (avoid critical)
sintheta = linspace(0, sin((pi/180)*thetamax), nthetas);	% fan of rays, equal in sin(theta)

%  Make arrays of replicated logs, dimensioned by the number of rays to shoot
    hh   = repmat(h,  1, nthetas);
    vvp  = repmat(vp, 1, nthetas);
    vvs  = repmat(vs, 1, nthetas);
    QQp  = repmat(Qp, 1, nthetas);
    QQs  = repmat(Qs, 1, nthetas);


%  Make arry of starting P-wave angles at the surface, dimensioned by number of layers
    sstt = repmat(sintheta, length(vp), 1);  

%  Compute array of downgoing P-wave "sin(theta)" using snell's law.  Results are actual
%  ray traced angles for each layer and each ray.
    ss = sstt.*vvp./vp(1);          % sin(theta)
	ss(ss>sinlim) = sinlim;         % DONT LET IT GO CRITICAL.  STOP RAY BENDING
	ss(ss<0) = 0;                   % DONT LET IT GO CRITICAL.  STOP RAY BENDING
    tt = ss./sqrt(1 - ss.^2);       % tan(theta)

%  Compute array of dx - increments of traveled distance in horizontal direction
%  within each layer, for each downgoing P-wave ray
    dx = hh.*tt;  % one-way horizontal propagation distance

%  Compute arry of dt - increment of travel time within each layer, for each 
%  down-going P-wave ray
    dt = sqrt(hh.^2 + dx.^2)./vvp;   % one-way time
    
%  Compute aray of Q-amplitude loss factor within each layer, for each ray
    dAp = dt./(QQp.*deltat);         % one-way amplitude loss in layer will be exp(-pi*dA)
                                     % dA=T/Q,  where T is travel time in layer in units of deltat
    dAp(1,:) = dAp(1,:) - dAp(1,1);  % normalize overburden effect relative to normal incidence
                                     % Hence, the input wavelet appears at the top of the model.

%  Make arry of starting S-wave angles at the BOTTOM, computed using Snell's law
%  and the ending downgoing P-wave angle at the bottom
    ssttps = repmat(ss(end,:).*vs(end)./vp(end), length(vp), 1);  

%  Compute array of upgoing S-wave "sin(theta)" using snell's law.  Results are actual
%  ray traced angles for each layer and each ray.
    ssps = ssttps.*vvs./vs(end);          % sin(theta)
	ssps(ssps>sinlim) = sinlim;           % DONT LET IT GO CRITICAL.  STOP RAY BENDING
	ssps(ssps<0) = 0;                     % DONT LET IT GO CRITICAL.  STOP RAY BENDING
    ttps = ssps./sqrt(1 - ssps.^2);       % tan(theta)

%  Compute array of dx - increments of traveled distance in horizontal direction
%  within each layer, for each ray
    dxps = hh.*ttps;  % one-way horizontal propagation distance

%  Compute arry of dt - increment of travel time within each layer, for each ray
    dtps = sqrt(hh.^2 + dxps.^2)./vvs;  % one-way time
    
%  Compute aray of Q-amplitude loss factor within each layer, for each ray
    dAs = dtps./(QQs.*deltat);       % one-way amplitude loss in layer will be exp(-pi*dA)
                                     % dA=T/Q,  where T is travel time in layer in units of deltat
    dAs(1,:) = dAs(1,:) - dAs(1,1);  % normalize overburden effect relative to normal incidence
                                     % Hence, the input wavelet appears at the top of the model.

%  Sum to get total travel distance and two-way time
    xcum = cumsum(dx)  + cumsum(dxps);
    tcum = cumsum(dt)  + cumsum(dtps);
    acum = cumsum(dAp) + cumsum(dAs);

% Compute vertical travel time
t0    = cumsum(h./vp) + cumsum(h./vs);  % Normal incidence time, unequally spaced.
ttop  = t0(1);                          % Time to top of model
tbot  = t0(end);                        % Time to bottom of model

% Convert layer properties from depth to equally spaced in time.  Logs are now time converted.
% Logs are first oversampled for interpolation, then blocked. 
dtavg = 1.0*mean(h(2:end)./vp(2:end));               %half average travel time through the model layers
nover = ceil(deltat/dtavg);            % oversample factor to take wavelet sample to dtavg
nover = max(1,nover) ;                 % oversample factor
time  = [ttop: deltat/nover: tbot]';   % Vector of equally sampled time, finely spaced
% vp, vs, and rho are interpolated with special function that references 'next' increment, 
% since input quantities are referenced to layer bottoms.
vpt    = interpnext(t0, vp,   time);  
vst    = interpnext(t0, vs,   time);  
rhot   = interpnext(t0, rho,  time); 
sst    = interp1(t0,    ss,   time);  
tcumt  = interp1(t0,    tcum, time);  
xcumt  = interp1(t0,    xcum, time);
acumt  = interp1(t0,    acum, time);

if nover >1, 
	vpt   = blockav(1./vpt, nover);  vpt = 1./vpt(1:nover:end);
	vst   = blockav(1./vst, nover);  vst = 1./vst(1:nover:end);
	rhot  = blockav(rhot, nover);   rhot = rhot(1:nover:end);
	sst   = blockav(sst, nover);     sst = sst(1:nover:end, :);
	tcumt = tcumt(1:nover:end, :);                              
	xcumt = xcumt(1:nover:end, :);                          
    acumt = acumt(1:nover:end, :);
end;
% quantities are now in layers equally sampled in vertical time.  
% First time sample tcumt is ray time to top of model
% First entry in xcumt is offset to top of model

% now compute zoeppritz at each layer boundary
rflag = diff(vpt)==0 & diff(vst)==0 & diff(rhot)==0;   % flag for layers with no contrast
lenrpst = 2*length(vpt);
Rpst = zeros(lenrpst, nx);                             % initialize equally spaced Rpp to zero; first row at top of model
Rpstnmo = Rpst;                                        % moved out reflectivities
ssxx    = nan*Rpst;                                    % angles
aaxx    = Rpst;
for i = 1:length(vpt)-1,                               % loop over layers
	if ~rflag(i),                                      % only compute at boundaries with contrast
	    ttx = interp1(xcumt(i+1,:), tcumt(i+1,:), x);  % 2-way time to this reflector, each desired offset
	    ssx = interp1(xcumt(i,:), sst(i,:), x);        % sintheta at this reflector, each offset
        ssxx(i,:) = ssx;
        aax = interp1(xcumt(i+1,:), acumt(i+1, :), x);
	    R= avops(vpt(i), vst(i), rhot(i), vpt(i+1), vst(i+1), rhot(i+1), (180/pi)*asin(ssx), 1);
        R(ssx >= sinlim-.02) = 0.;                      % mute at maximum allowed sin(theta)
%        R(~isreal(R)) = 0.; 
        R=real(R);                  % mute if reflectivity is becoming complex
        for j = 1:nx
            a = real(qfiltgm(1/aax(j),9));
            lena = length(a);
            Rpst(i+1:i+lena,j) = Rpst(i+1:i+lena,j) + a(:)*R(j);
        end;
%       find integer time step closest to moveout time ttx
%       nmofac gives weight, maximum when ttx falls right on integer time step
		nmo = max(1,min(1+floor((ttx-ttop)/deltat), lenrpst-lena+1)); nmofac = abs(nmo-(ttx-ttop)/deltat);
		nmonext = max(1,min(nmo+1,lenrpst-lena+1));
		for j=1:nx,     % moved out reflectivity is divided between two nearest time steps
            a = real(qfiltgm(1/aax(j),7));
            lena = length(a);
			Rpstnmo(nmo(j):nmo(j)+lena-1,         j) = Rpstnmo(nmo(j):nmo(j)+lena-1,         j) + a(:)*R(j)*nmofac(j);
			Rpstnmo(nmonext(j):nmonext(j)+lena-1, j) = Rpstnmo(nmonext(j):nmonext(j)+lena-1, j) + a(:)*R(j)*(1-nmofac(j));
		end; 
	end;
end;
Rpst    = Rpst(1:40+length(vpt),:);
Rpstnmo = Rpstnmo(1:40+length(vpt),:);
Rpst(isnan(Rpst)) = 0; Rpstnmo(isnan(Rpstnmo))=0;
Rpst(end, :) = 0;      Rpstnmo(end, :) = 0;
ssxx    = ssxx(1:40+length(vpt),:);
for k = 1:nx, ssxx(:,k)    = fillnan(ssxx(:,k)); end;

% convolve with wavelet
for k = 1:nx
	s(:,k) = conv(Rpst(:,k), wvlt);
	snmo(:,k) = conv(Rpstnmo(:,k), wvlt);
end;
s(1:nshift, :) = [];         % apply shift, if wavelet is non-causal
snmo(1:nshift, :) = [];         % apply shift, if wavelet is non-causal

if nargout == 0, 
  figure;seisplot(s,ttop,deltat,0, deltax)
  figure;seisplot(snmo,ttop,deltat,0, deltax)
end;
 
