function [s, snmo, ttop, x, ssxx] = rtcdpps(z, vp, vs, rho, deltax, nx, wvlt, deltat, nshift);
%        [s, snmo, ttop, x, ssxx] = rtcdpps(z, vp, vs, rho, deltax, nx, wvlt, deltat, nshift);
%
% Computes P-S cdp gather via raytracing in a 1-D Earth model.  Travel times,
% angles and offsets are computed via raytracing.  Amplitudes at each
% interface
% are computed using Zoeppritz.  Final reflectivity series is convolved with
% a wavelet.  
% *****  we recommend that the wavelet be sampled at least 15 samples per wavelength, 
%        in order to ensure that the depth-to-time converted Earth model is sufficiently
%        well sampled.
% Calculation is for precritical only -- rays are prevented from critical
%
% Inputs
%    z			Depths to BOTTOM of layers.  First z is thickness of first layer.
%               Can also be thought of as depth to reflectors.  First z is depth 
%               to first reflector. Computation starts from first depth point.
%	 vp,vs,rho	Interval properties corresponding to the depths
%    deltax     offset interval of desired traces.  First trace is normal incidence
%    nx         number of traces to compute
%    wvlt		input wavelet
%    deltat     sample interval of input wavelet and output seismogram (in seconds)
%    nshift     number of samples to shift, if wavelet is two-sided.  normally floor(length(wvlt)/2)
%
% Outputs
%    s          array of seismograms, one per column, moveout corrected
%    snmo       array of seismograms, one per column, with moveout
%    ttop       2-way time to top of model
%    xout       computed offsets, should be approximately equal to the input x
%    sst        array of sin(theta), giving angles of incidence
%               corresponding to each seismogram


% written by Gary Mavko

% ensure that log inputs are column vectors
z   = z(:);
vp  = vp(:);
vs  = vs(:);
rho = rho(:);

% get rid of nonunique depth points
[zzz,j]=unique(z);
z1 = z(1); z = z(j); z(1)=z1;
vp1 = vp(1); vp = vp(j); vp(1) = vp1;    % restores first velocity, incase nonuniqueness causes us to lose it
vs1 = vs(1); vs = vs(j); vs(1) = vs1;
rho1 = rho(1); rho = rho(j); rho(1) = rho1;

% compute interval thicknesses.  First layer extends from surface to first log depth point.
h(1) = z(1); h(2: length(z)) = diff(z); h = h(:);

% set up array of desired offsets for computed traces
x = [0: deltax: (nx-1)*deltax]; 

% Form an initial vector of starting ray angles, ranging from zero, to a max
% corresponding to a little past that corresponding to a straight ray.
% This initial fan of rays will be interpolated after ray tracing to get desired ray offsets.
nthetas = 25;                      % initial number of ray fans to shoot
thetamax = 50;                     % angle in degrees of maximum ray to shoot
sinlim = 0.98;                     % maximum allowed sin(theta) during ray tracing (avoid critical)
sintheta = linspace(0, sin((pi/180)*thetamax), nthetas);	% fan of rays, equal in sin(theta)

%  Make arrays of replicated logs, dimensioned by the number of rays to shoot
    hh   = repmat(h,  1, nthetas);
    vvp  = repmat(vp, 1, nthetas);
    vvs  = repmat(vs, 1, nthetas);

%  Make arry of starting P-wave angles at the surface, dimensioned by number of layers
    sstt = repmat(sintheta, length(vp), 1);  

%  Compute array of downgoing P-wave "sin(theta)" using snell's law.  Results are actual
%  ray traced angles for each layer and each ray.
    ss = sstt.*vvp./vp(1);          % sin(theta)
	ss(ss>sinlim) = sinlim;         % DONT LET IT GO CRITICAL.  STOP RAY BENDING
	ss(ss<0) = 0;                   % DONT LET IT GO CRITICAL.  STOP RAY BENDING
    tt = ss./sqrt(1 - ss.^2);       % tan(theta)

%  Compute array of dx - increments of traveled distance in horizontal direction
%  within each layer, for each downgoing P-wave ray
    dx = hh.*tt;  % one-way horizontal propagation distance

%  Compute arry of dt - increment of travel time within each layer, for each 
%  down-going P-wave ray
    dt = sqrt(hh.^2 + dx.^2)./vvp;  % one-way time
	
%  Make arry of starting S-wave angles at the BOTTOM, computed using Snell's law
%  and the ending downgoing P-wave angle at the bottom
    ssttps = repmat(ss(end,:).*vs(end)./vp(end), length(vp), 1);  

%  Compute array of upgoing S-wave "sin(theta)" using snell's law.  Results are actual
%  ray traced angles for each layer and each ray.
    ssps = ssttps.*vvs./vs(end);          % sin(theta)
	ssps(ssps>sinlim) = sinlim;           % DONT LET IT GO CRITICAL.  STOP RAY BENDING
	ssps(ssps<0) = 0;                     % DONT LET IT GO CRITICAL.  STOP RAY BENDING
    ttps = ssps./sqrt(1 - ssps.^2);       % tan(theta)

%  Compute array of dx - increments of traveled distance in horizontal direction
%  within each layer, for each ray
    dxps = hh.*ttps;  % one-way horizontal propagation distance

%  Compute arry of dt - increment of travel time within each layer, for each ray
    dtps = sqrt(hh.^2 + dxps.^2)./vvs;  % one-way time

%  Sum to get total travel distance and two-way time
    xcum = cumsum(dx) + cumsum(dxps);
    tcum = cumsum(dt) + cumsum(dtps);

% Compute vertical travel time
t0    = cumsum(h./vp) + cumsum(h./vs);  % Normal incidence time, unequally spaced.
ttop  = t0(1);                          % Time to top of model
tbot  = t0(end);                        % Time to bottom of model

% Convert layer properties from depth to equally spaced in time.  Logs are now time converted.
% Logs are first oversampled for interpolation, then blocked. 
dtavg = 1.0*mean(h(2:end)./vp(2:end));               %half average travel time through the model layers
nover = ceil(deltat/dtavg);            % oversample factor to take wavelet sample to dtavg
nover = max(1,nover) ;                 % oversample factor
time  = [ttop: deltat/nover: tbot]';   % Vector of equally sampled time, finely spaced
% vp, vs, and rho are interpolated with special function that references 'next' increment, 
% since input quantities are referenced to layer bottoms.
vpt    = interpnext(t0, vp,   time);  
vst    = interpnext(t0, vs,   time);  
rhot   = interpnext(t0, rho,  time); 
sst    = interp1(t0,    ss,   time);  
tcumt  = interp1(t0,    tcum, time);  
xcumt  = interp1(t0,    xcum, time);

if nover >1, 
	vpt   = blockav(1./vpt, nover);  vpt = 1./vpt(1:nover:end);
	vst   = blockav(1./vst, nover);  vst = 1./vst(1:nover:end);
	rhot  = blockav(rhot, nover);   rhot = rhot(1:nover:end);
	sst   = blockav(sst, nover);     sst = sst(1:nover:end, :);
	tcumt = tcumt(1:nover:end, :);                              
	xcumt = xcumt(1:nover:end, :);                          
end;
% quantities are now in layers equally sampled in vertical time.  
% First time sample tcumt is ray time to top of model
% First entry in xcumt is offset to top of model

% now compute zoeppritz at each layer boundary
rflag = diff(vpt)==0 & diff(vst)==0 & diff(rhot)==0;   % flag for layers with no contrast
lenrpst = 2*length(vpt);
Rpst = zeros(lenrpst, nx);                             % initialize equally spaced Rpp to zero; first row at top of model
Rpstnmo = Rpst;                                        % moved out reflectivities
ssxx    = nan*Rpst;                                    % angles
for i = 1:length(vpt)-1,                               % loop over layers
	if ~rflag(i),                                      % only compute at boundaries with contrast
	    ttx = interp1(xcumt(i+1,:), tcumt(i+1,:), x);  % 2-way time to this reflector, each desired offset
	    ssx = interp1(xcumt(i,:), sst(i,:), x);        % sintheta at this reflector, each offset
        ssxx(i,:) = ssx;
	    R = avops(vpt(i),   vst(i),   rhot(i), vpt(i+1), vst(i+1), rhot(i+1), (180/pi)*asin(ssx), 1);
        R(ssx >= sinlim-.02) = 0.;                      % mute at maximum allowed sin(theta)
%        R(~isreal(R)) = 0.; 
        R=real(R);                  % mute if reflectivity is becoming complex
        Rpst(i+1,:)=R;
%       find integer time step closest to moveout time ttx
%       nmofac gives weight, maximum when ttx falls right on integer time step
		nmo = max(1,min(1+floor((ttx-ttop)/deltat), lenrpst)); nmofac = abs(nmo-(ttx-ttop)/deltat);
		nmonext = max(1,min(nmo+1,lenrpst));
		for j=1:nx,     % moved out reflectivity is divided between two nearest time steps
			Rpstnmo(nmo(j),     j) = Rpstnmo(nmo(j),     j) + Rpst(i+1,j)*nmofac(j);
			Rpstnmo(nmonext(j), j) = Rpstnmo(nmonext(j), j) + Rpst(i+1,j)*(1-nmofac(j));
		end; 
	end;
end;
Rpst    = Rpst(1:40+length(vpt),:);
Rpstnmo = Rpstnmo(1:40+length(vpt),:);
Rpst(isnan(Rpst)) = 0; Rpstnmo(isnan(Rpstnmo))=0;
Rpst(end, :) = 0;      Rpstnmo(end, :) = 0;
ssxx    = ssxx(1:40+length(vpt),:);
for k = 1:nx,                  % fill nans resulting from interpolation
    if sum(~isnan(ssxx(:,k)))==0, ssxx(:,k)=.8; end;
    ssxx(:,k) = fillnan(ssxx(:,k)); 
end;

% convolve with wavelet
for k = 1:nx
	s(:,k) = conv(Rpst(:,k), wvlt);
	snmo(:,k) = conv(Rpstnmo(:,k), wvlt);
end;
s(1:nshift, :) = [];         % apply shift, if wavelet is non-causal
snmo(1:nshift, :) = [];         % apply shift, if wavelet is non-causal

%if nargout == 0, 
%  figure;seisplot(s,ttop,deltat,0, deltax)
%  figure;seisplot(snmo,ttop,deltat,0, deltax)
%end;
 
%---------------------
function Rps=avops(vp1,vs1,d1,vp2,vs2,d2,ang,approx);
%Rps=AVOPS(vp1,vs1,d1,vp2,vs2,d2,ang,approx);
%
%Calculates P-to-S reflectivity (Rps) as a function of
%the angle of incidence (ang).
%input parameters:
%  layer 1 (top): vp1, vs1, density1 (d1)
%  layer 2 (bottom): vp2, vs2, density2 (d2)
% ang: vector with angles(DEG)
% approx: 1)Full Zoeppritz (A&R)
%	  2)Aki & Richards
%         3)Donati's 98 SEG paper (quadratic)
%         4)Donati's 98 SEGpaper (linear)
%         5)"max" (linear) simplification
%	  6)Ezequiel Gonzalez' approx
%         7)Alejandro & Reinaldo approx
%
% With no output arguments, plots Rps vs. angle.
%

t=ang.*pi./180;	p=sin(t)./vp1;	ct=cos(t);
da=(d1+d2)/2;     Dd=(d2-d1);
vpa=(vp1+vp2)/2;  Dvp=(vp2-vp1);
vsa=(vs1+vs2)/2;  Dvs=(vs2-vs1);
cj1=sqrt(1-(sin(t).^2.*(vs1.^2./vp1.^2)));

switch approx
   case 1,		%FULL Zoeppritz (A&K)
	ct2=sqrt(1-(sin(t).^2.*(vp2.^2./vp1.^2)));
	cj2=sqrt(1-(sin(t).^2.*(vs2.^2./vp1.^2)));
	a=(d2.*(1-(2.*vs2.^2.*p.^2)))-(d1.*(1-(2.*vs1.^2.*p.^2)));
	b=(d2.*(1-(2.*vs2.^2.*p.^2)))+(2.*d1.*vs1.^2.*p.^2);
	c=(d1.*(1-(2.*vs1.^2.*p.^2)))+(2.*d2.*vs2.^2.*p.^2);
	d=2.*((d2.*vs2.^2)-(d1.*vs1.^2));
	E=(b.*ct./vp1)+(c.*ct2./vp2);
	F=(b.*cj1./vs1)+(c.*cj2./vs2);
	G=a-(d.*ct.*cj2./(vp1.*vs2));
	H=a-(d.*ct2.*cj1./(vp2.*vs1));
	D=(E.*F)+(G.*H.*p.^2);
	Rps=-2.*(ct./vp1).*...
		((a.*b)+(c.*d.*ct2.*cj2./(vp2.*vs2))).*p.*vp1./(vs1.*D);
   case 2,		%Aki & Richard (aprox)
%assuming (angles) i=i1, and j=j1
	Rps=(-p.*vpa./(2.*cj1)).*( ((Dd./da).*(1-(2.*vsa.^2.*p.^2)+ ...
		(2.*vsa.^2.*ct.*cj1./(vpa.*vsa)))) - ((Dvs./vsa).* ...
		((4.*vsa.^2.*p.^2)-(4.*vsa.^2.*ct.*cj1./(vpa.*vsa)))) );
   case 3,		%Donati's paper (quadratic)
	A0=-0.5.*(((Dd./da).*(1-(2.*vsa.^2./vpa.^2)))-(4.*vsa.*Dvs./vpa.^2));
	A1=-0.5.*(((Dd./da)+(2.*Dvs./vsa)).*((2.*vsa./vpa)-(vsa.^3./vpa.^3)));
	A2=-(vsa.^2./vpa.^2).*((Dd./da)+(2.*Dvs./vsa));
	Rps=sin(t).*(A0 + (A1.*ct) + (A2.*ct.^2));
   case 4,		%Donati's paper (linear)
	A0=-0.5.*(((Dd./da).*(1-(2.*vsa.^2./vpa.^2)))-(4.*vsa.*Dvs./vpa.^2));
	A1=-0.5.*(((Dd./da)+(2.*Dvs./vsa)).*((2.*vsa./vpa)-(vsa.^3./vpa.^3)));
	Rps=sin(t).*(A0 + (A1.*ct));
   case 5,		%Max. simplification
	Rps=-sin(t).*((Dd./(2.*da))-(((Dd./da)+(2.*Dvs./vsa)).*vsa/vpa));
   case 6,              %Ezequiel
	Rps=sin(t).*((-0.5.*Dd./da)-((vsa./vpa).*((Dd./da)+(2.*Dvs./vsa)))+...
	(((vsa./vpa).^3).*((0.5.*Dd./da)+(Dvs./vsa))));
    case 7,      %Alejandro & Reinaldo
        Rps=-2.*(vs1./vp1).*sin(t).*((Dd./da.*(0.5+(0.25.*vpa./vsa)))+...
        (Dvs./vsa));
   otherwise,	
end

if nargout==0
plot(ang,Rps)
end;
