function [s, snmo, ttop, x, ssxx] = rtcdpppQ(z, vp, vs, rho, deltax, nx, wvlt, deltat, nshift, Q, tmode);
%        [s, snmo, ttop, x, ssxx] = rtcdpppQ(z, vp, vs, rho, deltax, nx, wvlt, deltat, nshift, Q, tmode);
%
% Computes synthetic CMP gather via raytracing in a 1-D Earth model.  Travel times,
% angles and offsets are computed via raytracing.  Reflectivities are computed using 
% Zoeppritz, and trace comes from convolving with the wavelet.
%
% The input model is converted to equal sample intervals in time, equal to the time 
% interval of the input wavelet.  This obviously resamples model. The ray tracing 
% times, horizontal distance of propagation, and ray angles are computed on the original 
% model, and only the reflectivities are computed after resampling to time.
% Resampling is done carefully -- if layers are thin, then there is first oversampling
% at least once per layer, then velocities are averaged using Backus, and density 
% and angles are averaged linearly back to the wavelet sample interval.
%
% Ray geometries and times are computed on a broad fan of rays shot from the surface.  
% Reflectivities are computed at the desired offsets by interpolating angles and times 
% from the fan.  Reflectivities and transmission coefficients are computed using Zoeppritz 
% at the resampled layers, equally spaced in time.  
% Two reflectivity sequences are computed:  one with moveout and the other without.
% Then convolution creates trace.  Hence, there is no NMO stretch.
% 
% Calculation is for precritical only -- rays are prevented from critical
% *****  we recommend that the wavelet be sampled at least 15 samples per wavelength, 
%        in order to ensure that the depth-to-time converted Earth model is sufficiently
%        well sampled.
%
% Attenuation is estimated by replacing each reflection coefficent by a
% minimum phase attenuation filter that simulates the cumulative loss to
% that point.  An important caution:  since attenuation changes the
% wavelet, the input wavelet is defined as that entering the top of the
% model at the near-offset trace.
%
% Inputs
%    z			Depths to BOTTOM of layers.  First z is thickness of first layer.
%               Can also be thought of as depth to reflectors.  First z is depth 
%               to first reflector. Computation starts from first depth point.
%	 vp,vs,rho	Interval properties corresponding to the depths
%    deltax     offset interval of desired traces.  First trace is normal incidence
%    nx         number of traces to compute
%    wvlt		input wavelet
%    deltat     sample interval of input wavelet and output seismogram (in seconds)
%    nshift     number of samples to shift, if wavelet is two-sided.  normally floor(length(wvlt)/2)
%    Q          Q-value.  Scalar, or vector of length of z
%    tmode      =1 (default) to calculate transmission losses; =0 to ignore transmission losses
%
% Outputs
%    s          array of seismograms, one per column, moveout corrected
%    snmo       array of seismograms, one per column, with moveout
%    ttop       2-way time to top of model
%    x          vector of computed offsets
%    ssxx       array of sin(theta), giving angles of incidence
%               corresponding to each seismogram

% written by Gary Mavko; Updated March 2003.

if nargin<11, tmode =1; end;   % default is to include transmission losses

% ensure that log inputs are column vectors
z   = z(:);
vp  = vp(:);
vs  = vs(:);
rho = rho(:);
Q   = Q(:);
if length(Q)==1, Q = Q*ones(size(vp)); end;

% get rid of nonunique depth points
[zzz,j]=unique(z);
z1 = z(1); z = z(j); z(1)=z1;
vp1 = vp(1); vp = vp(j); vp(1) = vp1;    % restores first velocity, incase nonuniqueness causes us to lose it
vs1 = vs(1); vs = vs(j); vs(1) = vs1;
rho1 = rho(1); rho = rho(j); rho(1) = rho1;
Q1 = Q(1); Q   = Q(j); Q(1) = Q1;

% compute interval thicknesses.  First layer extends from surface to first log depth point.
h(1) = z(1); h(2: length(z)) = diff(z); h = h(:);

% set up array of desired offsets for computed traces
x = [0: deltax: (nx-1)*deltax]; 

% Form an initial vector of starting ray angles, ranging from zero
% This initial fan of rays will be interpolated after ray tracing to get desired ray offsets.
nthetas = 25;                      % initial number of ray fans to shoot
thetamax = 50;                     % angle in degrees of maximum ray to shoot
sinlim = 0.98;                     % maximum allowed sin(theta) during ray tracing (avoid critical)
sintheta = linspace(0, sin((pi/180)*thetamax), nthetas);	% fan of rays, equal in sin(theta)

%  Make arrays of replicated logs, dimensioned by the number of rays to shoot
    hh   = repmat(h,  1, nthetas);
    vvp  = repmat(vp, 1, nthetas);
    QQ   = repmat(Q,  1, nthetas);

%  Make array of starting angles, dimensioned by number of layers
    sstt = repmat(sintheta, length(vp), 1);  

%  Compute array of "sin(theta)" using snell's law.  Results are actual
%  ray traced angles for each layer and each ray.
    ss = sstt.*vvp./vp(1);          % sin(theta)
	ss(ss>sinlim) = sinlim;         % DONT LET IT GO CRITICAL.  STOP RAY BENDING
	ss(ss<0)     = 0;               % DONT LET IT GO CRITICAL.  STOP RAY BENDING
    tt = ss./sqrt(1 - ss.^2);       % tan(theta)

%  Compute array of dx - increments of traveled distance in horizontal direction
%  within each layer, for each ray
    dx = hh.*tt;  % one-way horizontal propagation distance

%  Compute arry of dt - increment of travel time within each layer, for each ray
    dt = sqrt(hh.^2 + dx.^2)./vvp;  % one-way time

%  Compute aray of Q-amplitude loss factor within each layer, for each ray
    dA = dt./(QQ.*deltat);           % one-way amplitude loss in layer will be exp(-pi*dA)
                                     % dA=T/Q,  where T is travel time in layer in units of deltat
    dA(1,:) = dA(1,:) - dA(1,1);     % normalize overburden effect relative to normal incidence
                                     % Hence, the input wavelet appears at the top of the model.

%  Sum to get total travel distance and two-way time along each ray
    xcum = 2.*cumsum(dx);            % cumulative 2-way time
    tcum = 2.*cumsum(dt);            % cumulative 2-way horizontal distance
    acum = 2.*cumsum(dA);            % cumulative 2-way loss factor (T/Q)

% Compute vertical travel time
t0    = 2*cumsum(h./vp);              % Normal incidence time, unequally spaced.
ttop  = t0(1);                        % Time to top of model
tbot  = t0(end);                      % Time to bottom of model

% Convert layer properties from depth to equally spaced in time.  Logs are now time converted.
% Logs are first oversampled for interpolation, then blocked. 
dtavg = 1.0*mean(h(2:end)./vp(2:end)); % half average travel time through the model layers
nover = ceil(deltat/dtavg);            % oversample factor to take wavelet sample to dtavg
nover = max(1,nover) ;                 % oversample factor
time  = [ttop: deltat/nover: tbot]';   % Vector of equally sampled time, finely spaced

% convert all layer quantities to equally spaced in time.
% vp, vs, and rho are interpolated with special function that references 'next' increment, 
% since input quantities are referenced to layer bottoms.
vpt    = interpnext(t0, vp,   time);  
vst    = interpnext(t0, vs,   time);  
rhot   = interpnext(t0, rho,  time); 
sst    = interp1(t0,    ss,   time);  
tcumt  = interp1(t0,    tcum, time);  
xcumt  = interp1(t0,    xcum, time);
acumt  = interp1(t0,    acum, time);
if nover > 1  , 
	vpt   = blockav(1./vpt, nover);  vpt = 1./vpt(1:nover:end);
	vst   = blockav(1./vst, nover);  vst = 1./vst(1:nover:end);
	rhot  = blockav(rhot, nover);   rhot = rhot(1:nover:end);
	sst   = blockav(sst, nover);     sst = sst(1:nover:end, :);
	tcumt = tcumt(1:nover:end, :);                              
	xcumt = xcumt(1:nover:end, :);      
    acumt = acumt(1:nover:end, :);
end;
% quantities are now in layers equally sampled in vertical time.  
% First time sample tcumt is ray time to top of model
% First entry in xcumt is offset to top of model

% now compute zoeppritz at each layer boundary - equally spaced in time
rflag = diff(vpt)==0 & diff(vst)==0 & diff(rhot)==0;   % flag for layers with no contrast
lenrppt = 2*length(vpt);
Rppt = zeros(lenrppt, nx);      % initialize equally spaced Rpp to zero; first row at top of model
Rpptnmo = Rppt;                 % moved out reflectivities
ssxx    = nan*Rppt;             % angles
aaxx    = Rppt;
T       = ones(1,nthetas);      % initialize transmission coeffients for each ray
for i = 1:length(vpt)-1,                               % loop over layers
	if ~rflag(i),                                      % only compute at boundaries with contrast
	    ttx = interp1(xcumt(i+1,:), tcumt(i+1,:), x);  % 2-way time to this reflector, each desired offset
	    ssx = interp1(xcumt(i,:), sst(i,:), x);        % sintheta at this reflector, each offset
        ssxx(i,:) = ssx;
        aax = interp1(xcumt(i+1,:), acumt(i+1, :), x); % cumulative attenuation factor, each desired offset
	    R   = avopp(vpt(i), vst(i), rhot(i), vpt(i+1), vst(i+1), rhot(i+1), (180/pi)*asin(ssx), 1);
        R(ssx >= sinlim-.02) = 0.;                      % mute at maximum allowed sin(theta)
        R(~isreal(R)) = 0.; R=real(R);                  % mute if reflectivity is becoming complex
        if i > 1 & tmode==1,
           [rtmp,tdown] = avoppt(vpt(i-1),vst(i-1),rhot(i-1),vpt(i),vst(i),rhot(i),(180/pi)*asin(sst(i-1)), 1); % compute for each downgoing RAY
           [rtmp,tup  ] = avoppt(vpt(i),vst(i),rhot(i),vpt(i-1),vst(i-1),rhot(i-1),(180/pi)*asin(sst(i)), 1);   % compute for each  upgoaing RAY
           T = tdown.*tup.*T;                           % cumulative transmission, 
           Tx  = interp1(xcumt(i+1,:), T, x);           % interpolate transmission loss for each offset   
           R = R.*Tx;
        end;
        for j = 1:nx
            a = real(qfiltgm(1/aax(j),15));
%            a = real(qfiltgm(1/aax(j),9));
            lena = length(a);
            Rppt(i+1:i+lena,j) = Rppt(i+1:i+lena,j) + a(:)*R(j);
        end;
%       find integer time step closest to moveout time ttx
%       nmofac gives weight, maximum when ttx falls right on integer time step
		nmo = max(1,min(1+floor((ttx-ttop)/deltat), lenrppt-lena+1)); nmofac = abs(nmo-(ttx-ttop)/deltat);
		nmonext = max(1,min(nmo+1,lenrppt-lena+1));
		for j=1:nx,     % moved out reflectivity is divided between two nearest time steps
            a = real(qfiltgm(1/aax(j),9));
            lena = length(a);
			Rpptnmo(nmo(j):nmo(j)+lena-1,         j) = Rpptnmo(nmo(j):nmo(j)+lena-1,         j) + a(:)*R(j)*nmofac(j);
			Rpptnmo(nmonext(j):nmonext(j)+lena-1, j) = Rpptnmo(nmonext(j):nmonext(j)+lena-1, j) + a(:)*R(j)*(1-nmofac(j));
		end; 
	end;
end;
Rppt    = Rppt(1:40+length(vpt),:);
Rpptnmo = Rpptnmo(1:40+length(vpt),:);
Rppt(isnan(Rppt)) = 0; Rpptnmo(isnan(Rpptnmo))=0;
Rppt(end, :) = 0;      Rpptnmo(end, :) = 0;
ssxx    = ssxx(1:40+length(vpt),:);
for k = 1:nx, ssxx(:,k)    = fillnan(ssxx(:,k)); end;

% convolve with wavelet
for k = 1:nx
	s(:,k)    = conv(Rppt(:,k), wvlt);
	snmo(:,k) = conv(Rpptnmo(:,k), wvlt);
end;
s(1:nshift, :) = [];         % apply shift, if wavelet is non-causal
snmo(1:nshift, :) = [];      % apply shift, if wavelet is non-causal


if nargout == 0, 
 figure;seiswigrwb(s,ttop,deltat,0, deltax)
 figure;seisplot(s,ttop,deltat,0, deltax)
 figure;seiswigrwb(snmo,ttop,deltat,0, deltax)
end;
 
