function [s, snmo, ttop, x, ssxx, ttxx, tcumt, xcumt] = rtcdppp(z, vp, vs, rho, deltax, nx, wvlt, deltat, nshift, tmode);
%        [s, snmo, ttop, x, ssxx, ttxx, tcumt, xcumt] = rtcdppp(z, vp, vs, rho, deltax, nx, wvlt, deltat, nshift, tmode);
%
% Computes synthetic CMP gather via raytracing in a 1-D Earth model.  Travel times,
% angles and offsets are computed via raytracing.  Reflectivities are computed using 
% Zoeppritz, and trace comes from convolving with the wavelet.
%
% The input model is converted to equal sample intervals in time, equal to the time 
% interval of the input wavelet.  This obviously resamples model. The ray tracing 
% times, horizontal distance of propagation, and ray angles are computed on the original 
% model, and only the reflectivities are computed after resampling to time.
% Resampling is done carefully -- if layers are thin, then there is first oversampling
% at least once per layer, then velocities are averaged using Backus, and density 
% and angles are averaged linearly back to the wavelet sample interval.
%
% Ray geometries and times are computed on a broad fan of rays shot from the surface.  
% Reflectivities are computed at the desired offsets by interpolating angles and times 
% from the fan.  Reflectivities and transmission coefficients are computed using Zoeppritz 
% at the resampled layers, equally spaced in time.  
% Two reflectivity sequences are computed:  one with moveout and the other without.
% Then convolution creates trace.  Hence, there is no NMO stretch.
% 
% Calculation is for precritical only -- rays are prevented from critical
% *****  we recommend that the wavelet be sampled at least 15 samples per wavelength, 
%        in order to ensure that the depth-to-time converted Earth model is sufficiently
%        well sampled.
%
% Inputs
%    z			Depths to BOTTOM of layers.  First z is thickness of first layer.
%               Can also be thought of as depth to reflectors.  First z is depth 
%               to first reflector. Computation starts from first depth point.
%	 vp,vs,rho	Interval properties corresponding to the depths
%    deltax     offset interval of desired traces.  First trace is normal incidence
%    nx         number of traces to compute
%    wvlt		input wavelet
%    deltat     sample interval of input wavelet and output seismogram (in seconds)
%    nshift     number of samples to shift, if wavelet is two-sided.  normally floor(length(wvlt)/2)
%    tmode      =1 (default) to calculate transmission losses; =0 to ignore transmission losses
%
% Outputs
%    s          array of seismograms, one per column, moveout corrected
%    snmo       array of seismograms, one per column, with moveout
%    ttop       2-way time to top of model
%    x          vector of computed offsets
%    ssxx       array of sin(theta), giving angles of incidence
%               corresponding to each time step in the seismogram at each recorded offset
%    ttxx       array of two-way times, corresponding to each time step in the seismogram
%               at each recorded offset
%
% calls functions:  blockav,interpnext,avoppt

% written by Gary Mavko; Updated March 2003.

if nargin<10, tmode =1; end;   % default is to include transmission losses

% ensure that log inputs are column vectors
z   = z(:);   
vp  = vp(:);
vs  = vs(:);
rho = rho(:);

% get rid of nonunique depth points
[zzz,j]=unique(z); 
z1 = z(1); z = z(j); z(1)=z1;
vp1 = vp(1); vp = vp(j); vp(1) = vp1;    % restores first velocity, incase nonuniqueness causes us to lose it
vs1 = vs(1); vs = vs(j); vs(1) = vs1;
rho1 = rho(1); rho = rho(j); rho(1) = rho1;

% compute interval thicknesses.  First layer extends from surface to first log depth point.
h(1) = z(1); h(2: length(z)) = diff(z); h = h(:);

% set up array of desired offsets for computed traces
x = [0: deltax: (nx-1)*deltax]; 

% Form an initial vector of starting ray angles, ranging from zero
% This initial fan of rays will be interpolated after ray tracing to get desired ray offsets.
nthetas = 25;                      % initial number of ray fans to shoot
thetamax = 50;                     % angle in degrees of maximum ray to shoot
sinlim = 0.98;                     % maximum allowed sin(theta) during ray tracing (avoid critical)
sintheta = linspace(0, sin((pi/180)*thetamax), nthetas);	% fan of rays, equal in sin(theta)

%  Make arrays of replicated logs, dimensioned by the number of rays to shoot
    hh   = repmat(h,  1, nthetas);
    vvp  = repmat(vp, 1, nthetas);

%  Make array of starting angles, dimensioned by number of layers
    sstt = repmat(sintheta, length(vp), 1);  

%  Compute array of "sin(theta)" using snell's law.  Results are actual
%  ray traced angles for each layer and each ray.
    ss = sstt.*vvp./vp(1);          % sin(theta)
	ss(ss>sinlim) = sinlim;         % DONT LET IT GO CRITICAL.  STOP RAY BENDING
	ss(ss<0)     = 0;               % DONT LET IT GO CRITICAL.  STOP RAY BENDING
    tt = ss./sqrt(1 - ss.^2);       % tan(theta)

%  Compute array of dx - increments of traveled distance in horizontal direction
%  within each layer, for each ray
    dx = hh.*tt;  % one-way horizontal propagation distance

%  Compute arry of dt - increment of travel time within each layer, for each ray
    dt = sqrt(hh.^2 + dx.^2)./vvp;  % one-way time

%  Sum to get total travel distance and two-way time along each ray
    xcum = 2.*cumsum(dx);
    tcum = 2.*cumsum(dt);

% Compute vertical travel time
t0    = 2*cumsum(h./vp);              % Normal incidence time, unequally spaced.
ttop  = t0(1);                        % Time to top of model
tbot  = t0(end);                      % Time to bottom of model

% Convert layer properties from depth to equally spaced in time.  Logs are now time converted.
% Logs are first oversampled for interpolation, then blocked. 
dtavg = 1.0*mean(h(2:end)./vp(2:end));               %half average travel time through the model layers
nover = ceil(deltat/dtavg);            % oversample factor to take wavelet sample to dtavg
nover = max(1,nover) ;                 % oversample factor
time  = [ttop: deltat/nover: tbot]';   % Vector of equally sampled time, finely spaced

% convert all layer quantities to equally spaced in time.
% vp, vs, and rho are interpolated with special function that references 'next' increment, 
% since input quantities are referenced to layer bottoms.
vpt    = interpnext(t0, vp,   time);  
vst    = interpnext(t0, vs,   time);  
rhot   = interpnext(t0, rho,  time); 
sst    = interp1(t0,    ss,   time);  
tcumt  = interp1(t0,    tcum, time);  
xcumt  = interp1(t0,    xcum, time);
if nover > 1  , 
	vpt   = blockav(1./vpt, nover);  vpt = 1./vpt(1:nover:end);
	vst   = blockav(1./vst, nover);  vst = 1./vst(1:nover:end);
	rhot  = blockav(rhot, nover);   rhot = rhot(1:nover:end);
	sst   = blockav(sst, nover);     sst = sst(1:nover:end, :);
	tcumt = tcumt(1:nover:end, :);                              
	xcumt = xcumt(1:nover:end, :);                          
end;
% quantities are now in layers equally sampled in vertical time.  
% First time sample tcumt is ray time to top of model
% First entry in xcumt is offset to top of model

% now compute zoeppritz at each layer boundary - equally spaced in time
rflag = diff(vpt)==0 & diff(vst)==0 & diff(rhot)==0;   % flag for layers with no contrast
lenrppt = max(2*length(vpt),length(vpt)+40);
Rppt = zeros(lenrppt, nx);                             % initialize equally spaced Rpp to zero; first row at top of model
Rpptnmo = Rppt;                                        % moved out reflectivities
ssxx    = nan*Rppt;                                    % angles
T       = ones(1,nthetas);                             % initialize transmission coeffients for each ray

for i = 1:length(vpt)-1,                               % loop over layers
	ttxx(i,:) = interp1(xcumt(i+1,:), tcumt(i+1,:), x);  % 2-way time to this reflector, each desired offset
	if ~rflag(i),                                      % only compute at boundaries with contrast
	    ttx = interp1(xcumt(i+1,:), tcumt(i+1,:), x);  % 2-way time to this reflector, each desired offset
	    ssx = interp1(xcumt(i,:), sst(i,:), x);        % sintheta at this reflector, each offset
        ssxx(i,:) = ssx;
        R = avoppt(vpt(i),   vst(i),   rhot(i), vpt(i+1), vst(i+1), rhot(i+1), (180/pi)*asin(ssx), 1);         % compute for each OFFSET
        R(ssx >= sinlim-.02) = 0.;                      % mute at maximum allowed sin(theta)
        R=real(R);                  % mute if reflectivity is becoming complex
        if i > 1 & tmode==1,
           [rtmp,tdown] = avoppt(vpt(i-1),vst(i-1),rhot(i-1),vpt(i),vst(i),rhot(i),(180/pi)*asin(sst(i-1)), 1); % compute for each downgoing RAY
           [rtmp,tup  ] = avoppt(vpt(i),vst(i),rhot(i),vpt(i-1),vst(i-1),rhot(i-1),(180/pi)*asin(sst(i)), 1);   % compute for each  upgoaing RAY
           T = tdown.*tup.*T;                           % cumulative transmission, 
           Tx  = interp1(xcumt(i+1,:), T, x);           % interpolate transmission loss for each offset   
           R = R.*Tx;
        end;
        Rppt(i+1,:)=R;
%       find integer time step closest to moveout time ttx
%       nmofac gives weight, maximum when ttx falls right on integer time step
		nmo = max(1,min(1+floor((ttx-ttop)/deltat), lenrppt)); nmofac = abs(nmo-(ttx-ttop)/deltat);
		nmonext = max(1,min(nmo+1,lenrppt));
		for j=1:nx,     % moved out reflectivity is divided between two nearest time steps
			Rpptnmo(nmo(j),     j) = Rpptnmo(nmo(j),     j) + Rppt(i+1,j)*nmofac(j);
			Rpptnmo(nmonext(j), j) = Rpptnmo(nmonext(j), j) + Rppt(i+1,j)*(1-nmofac(j));
		end; 
	end;
end;
Rppt    = Rppt(1:40+length(vpt),:);
Rpptnmo = Rpptnmo(1:40+length(vpt),:);
Rppt(isnan(Rppt)) = 0; Rpptnmo(isnan(Rpptnmo))=0;
Rppt(end, :) = 0;      Rpptnmo(end, :) = 0;
ssxx    = ssxx(1:40+length(vpt),:);
for k = 1:nx, ssxx(:,k)    = fillnan(ssxx(:,k)); end;

% convolve with wavelet
for k = 1:nx
	s(:,k)    = conv(Rppt(:,k), wvlt);
	snmo(:,k) = conv(Rpptnmo(:,k), wvlt);
end;
s(1:nshift, :) = [];         % apply shift, if wavelet is non-causal
snmo(1:nshift, :) = [];      % apply shift, if wavelet is non-causal


if nargout == 0, 
  figure;seisplot(s,ttop,deltat,0, deltax)
  figure;seisplot(snmo,ttop,deltat,0, deltax)
end;
 
%------------------------------------------------------------------
function  y=interpnext(x1, y1, x)
% y = interpnext(x1, y1, x)
%
% interpolation, similar to interp1, except that it uses the 'next' option.
% For desired new sample at x, the input function is sampled at the next sample x1
% greater or equal to x
% Method alters input data, by turning each point into a pair, so that normal
% interpolation is fooled.

dx = max(100*eps,mean(diff(x1))*1e-06);

x1(end) = max(x1(end),max(x) + dx);
y1(end+1) = y1(end);
n1 = length(x1);
n  = length(x);

xx1(1:2:2*n1-1) = x1;
xx1(2:2:2*n1)   = x1+dx;
yy1(1:2:2*n1-1) = y1(1:end-1);
yy1(2:2:2*n1)   = y1(2:end);

y = interp1(xx1,yy1,x);

%--------------------------------------------------------------------
function [logb]=blockav(log, nb)
%logb = blockav(log,nb)
%
%Block average of input LOG over NB points. LOG can be a single column
%vector or multiple column matrix with Nan as missing values. 
%LOG is padded by the last row if the total number of rows in LOG is 
%not an integer multiple of NB. Output is the blocked log LOGB with
%the same number of rows as LOG.

% Written by T. Mukerji, 1998

if size(log,1)==1,log=log(:); end; [nr,nc]=size(log);

npad=nb-rem(nr,nb);
if npad~=0, lrow=log(end,:); logpad=[log;lrow(ones(npad,1),:)]; end;

logpad=reshape(logpad,nb,(nr+npad)/nb,nc); logpad=nanmean(logpad);
logpad=reshape(logpad(ones(nb,1),:,:),nr+npad,nc);
logb=logpad(1:nr,:);


%---------------------------------------------------------------------
function [Rpp,Tpp]=avoppt(vp1,vs1,d1,vp2,vs2,d2,ang,approx);
%        [Rpp,Tpp]=avoppt(vp1,vs1,d1,vp2,vs2,d2,ang,approx);
%
% Calculates P-to-P reflectivity (Rpp) and transmission (Tpp) as a 
% function of the angle of incidence (ang).
% Input parameters:
%  layer 1 (top): vp1, vs1, density1 (d1)
%  layer 2 (bottom): vp2, vs2, density2 (d2)
%  ang: vector with incidence angles (DEGREES) 
%  approx: =1  Full Zoeppritz(A&R)
%	       =2  Aki&Richards
%          =3  Shuey's paper
%          =4  Castagna's paper->Shuey (slightly different formulation of Shuey)
%  --> For transmission, only approx =1 and 2 are implemented, so 3 and 4
%      Revert to Aki and Richards
%
% With no output arguments, plots Rpp vs. angle.
%

% Original written by Ezequiel Gonzalez (Oct,1999) at Stanford Rock Physics Lab
% Transmission coefficients added by Gary Mavko, March 2003

t=ang.*pi./180;	ct=cos(t);                   % vector of angles in radians
p=sin(t)./vp1;	                             % vector ray parameters, for each angle 
da=(d1+d2)/2;     Dd=(d2-d1);                % average and difference of density
vpa=(vp1+vp2)/2;  Dvp=(vp2-vp1);             % average and difference of Vp
vsa=(vs1+vs2)/2;  Dvs=(vs2-vs1);             % average and difference of Vs

switch approx
   case 1,		%FULL Zoeppritz (A&K)
	ct2=sqrt(1-(sin(t).^2.*(vp2.^2./vp1.^2)));
	cj1=sqrt(1-(sin(t).^2.*(vs1.^2./vp1.^2)));
	cj2=sqrt(1-(sin(t).^2.*(vs2.^2./vp1.^2)));
	a=(d2.*(1-(2.*vs2.^2.*p.^2)))-(d1.*(1-(2.*vs1.^2.*p.^2)));
	b=(d2.*(1-(2.*vs2.^2.*p.^2)))+(2.*d1.*vs1.^2.*p.^2);
	c=(d1.*(1-(2.*vs1.^2.*p.^2)))+(2.*d2.*vs2.^2.*p.^2);
	d=2.*((d2.*vs2.^2)-(d1.*vs1.^2));
	E=(b.*ct./vp1)+(c.*ct2./vp2);
	F=(b.*cj1./vs1)+(c.*cj2./vs2);
	G=a-(d.*ct.*cj2./(vp1.*vs2));
	H=a-(d.*ct2.*cj1./(vp2.*vs1));
	D=(E.*F)+(G.*H.*p.^2);
	Rpp=( (((b.*ct./vp1)-(c.*ct2./vp2)).*F) - ...
            ((a+(d.*ct.*cj2./(vp1.*vs2))).*H.*p.^2) ) ./ D;
    Tpp=2*d1.*(ct./vp1).*F.*(vp1./(vp2.*D));

   case 2,		%Aki & Richard (aprox), %assuming (angles) i=i1
	Rpp=(0.5.*(1-(4.*p.^2.*vsa.^2)).*Dd./da) + (Dvp./(2.*ct.^2.*vpa)) - ...
       	  (4.*p.^2.*vsa.*Dvs);
    Tpp=1-(0.5.*(Dd./da))+((1./(2.*ct.^2))-1).*(Dvp./vpa);

   case 3,		%Shuey
	poi1=((0.5.*(vp1./vs1).^2)-1)./((vp1./vs1).^2-1);
	poi2=((0.5.*(vp2./vs2).^2)-1)./((vp2./vs2).^2-1);
	poia=(poi1+poi2)./2;   Dpoi=(poi2-poi1);
	Ro=0.5.*((Dvp./vpa)+(Dd./da));
	Bx=(Dvp./vpa)./((Dvp./vpa)+(Dd./da));
	Ax=Bx-(2.*(1+Bx).*(1-2.*poia)./(1-poia));
	Rpp= Ro + (((Ax.*Ro)+(Dpoi./(1-poia).^2)).*sin(t).^2) + ...
	         (0.5.*Dvp.*(tan(t).^2-sin(t).^2)./vpa);
    Tpp=1-(0.5.*(Dd./da))+((1./(2.*ct.^2))-1).*(Dvp./vpa);  %Aki Richards approx for Tpp
   case 4,		%Shuey linear
	A=0.5.*((Dvp./vpa)+(Dd./da));
	B=(-2.*vsa.^2.*Dd./(vpa.^2.*da)) + (0.5.*Dvp./vpa) - ...
	  (4.*vsa.*Dvs./(vpa.^2));
	Rpp=A+(B.*sin(t).^2);
       Tpp=1-(0.5.*(Dd./da))+((1./(2.*ct.^2))-1).*(Dvp./vpa);  %Aki Richards approx for Tpp
otherwise,	
end

if nargout==0
plot(ang,Rpp)
end;
