function [S1,K1,S2,K2,S3,K3,S4,K4,S5,K5] = gibiansky_torquato_crossbounds(f1, f2, s1, s2, k1, k2, g1, g2)
% gibiansky_torquato_crossbounds(f1, f2, s1, s2, k1, k2, g1, g2)
%
% cross bounds between bulk modulus and electrical conductivity according
% to Gibiansky and Torquato (1996), 'Connection between the Conductivity 
% and Bulk Modulus of Isotropic Composite Materials
% Proceedings: Mathematical, Physical and Engineering Sciences, Vol. 452, 253-283
% Program plots 5 hyperbolas;  bounds are the outer two
%
% Inputs:
%    f1, f2       volume fractions of phase 1 and phase 2
%    s1, s2       conductivities of materials 1 and 2
%    k1, k2       bulk modulus of materials 1 and 2
%    g1, g2       shear modulus of materials 1 and 2
% Outputs
%    Si,Ki        conductivity-modulus pairs for each of the 5 Gibiansky & Torquato equations
%
% Example from G&T figure 1:
%             gibiansky_torquato_crossbounds(0.2, 0.8, 1, 20, 1, 20, 1/pr2kr(.3), 20/pr2kr(.3))

titstr=['f1=',num2str(f1,3),'; s1=',num2str(s1,3),'; s2=',num2str(s2,3),'; k1=',num2str(k1,3),'; k2=',num2str(k2,3),'; g1=',num2str(g1,3),'; g2=',num2str(g2,3)];

s1_star  =  F(s1, s2, f1, f2, 2*s1);
s2_star  =  F(s1, s2, f1, f2, 2*s2);
s1_hash  =  F(s1, s2, f1, f2, -2*s1);
s2_hash  =  F(s1, s2, f1, f2, -2*s2);

k1_star  =  F(k1, k2, f1, f2, (4/3)*g1);
k2_star  =  F(k1, k2, f1, f2, (4/3)*g2);

sa  = f1*s1 + f2*s2;
sh  = 1/(f1/s1 + f2/s2);
ka  = f1*k1 + f2*k2;
kh  = 1/(f1/k1 + f2/k2);

[S1, K1] = hyp(s1_star, k1_star, s2_star, k2_star, s1, k1);        % double coated sphere
[S2, K2] = hyp(s1_star, k1_star, s2_star, k2_star, s2, k2);        % double coated sphere
[S3, K3] = hyp(s1_star, k1_star, s2_star, k2_star, s1_hash, kh);   %
[S4, K4] = hyp(s1_star, k1_star, s2_star, k2_star, s2_hash, kh);
[S5, K5] = hyp(s1_star, k1_star, s2_star, k2_star, sa, ka);        % 3-point bounds

V1 = K2V(K1,f1);
V2 = K2V(K2,f1);
V3 = K2V(K3,f1);
V4 = K2V(K4,f1);
V5 = K2V(K5,f1);

if nargout==0,
    figure; hold on;
    plot(S1, K1, '-k','linewidth',2);
    plot(S2, K2, '-k','linewidth',2);
    plot(S3, K3, '--k');
    plot(S4, K4, '--k');
    plot(S5, K5, '-k');
    plot([s1_star, s2_star, s2_star, s1_star],[k1_star, k1_star, k2_star, k2_star],'k')
    ylabel('bulk modulus'), xlabel('conductivity');
    title(titstr);
    fsa16
    
%     figure; hold on;
%     plot(s2./S1, K1, '-k');
%     plot(s2./S2, K2, '-k');
%     plot(s2./S3, K3, '-k');
%     plot(s2./S4, K4, '-k');
%     plot(s2./S5, K5, '-k');
%     xlabel('F'); ylabel('bulk modulus');
%     title(titstr);
%     fsa16
%     
%     figure; hold on;
%     plot(s2./S1, V1, '-k');
%     plot(s2./S2, V2, '-k');
%     plot(s2./S3, V3, '-k');
%     plot(s2./S4, V4, '-k');
%     plot(s2./S5, V5, '-k');
%     xlabel('F'); ylabel('velocity');
%     title(titstr);
%     fsa16
    
end;

%---------------------------------------------------
function z = F( d1, d2, f1, f2, y) 
z = f1*d1 + f2*d2 - f1*f2*((d1-d2)^2)/(f2*d1+f1*d2+y);

%---------------------------------------------------
function [x, y] = hyp(x1, y1, x2, y2, x3, y3)
% Constructs the segment AB of a hyperbola that passes through the points
% A=(x1, y1), B=(x2, y2), and C = (x3, y3).

% paramter gam traces out curve by spanning 0<= gam <= 1
gam = linspace(0, 1, 100000);

% x,y values of hyperbola segment
x = gam*x1 + (1-gam)*x2 - gam.*(1-gam)*(x1 - x2)^2./((1-gam)*x1 + gam*x2 - x3);
y = gam*y1 + (1-gam)*y2 - gam.*(1-gam)*(y1 - y2)^2./((1-gam)*y1 + gam*y2 - y3);

%---------------------------------------------------
function Vp = K2V(K,f1)

G = K./pr2kr(.25);
rho = f1.*2.65 + (1-f1)*1;
Vp = sqrt((K+(4/3)*G)./rho);

