function [Condeff,R,f]=cond_iso_shells_scm(fracs1,conds1,fracs2,conds2,f1,f2)
% Effective electrical conductivity for multi-component ISOTROPIC composite
% where each SPHERICAL INCLUSION is a set of LAYERED SHELLS.  Strategy is to replace
% each unit of shells with a single effective conductivity and then proceed
% as with the usual spherical inclusion calculation.  
% This version allows for two different shell units, each with different
% materials, if desired.  
% ** can also be directly applied to DIELECTRIC CONSTANT instead of CONDUCTIVITY
%
% Effective ELECTRICAL CONDUCTIVITY for ISOTROPIC composite of CONCENTRIC SPHERICAL
% inclusions using the SELF CONSISTENT METHOD method (SCM) .
% This version allows multiple SPHERICAL constituents.
% Inputs:
%    fracs1      vector of volume fractions in FIRST type of multilayer inclusion, STARTING FROM THE OUTSIDE 
%    conds1      vector of phase conductivities in FIRST type of multilayer inclusion, STARTING FROM THE OUTSIDE
%    fracs2      vector of volume fractions in SECOND type of multilayer inclusion, STARTING FROM THE OUTSIDE 
%    conds2      vector of phase conductivities in SECOND type of multilayer inclusion, STARTING FROM THE OUTSIDE
%    f1,f2       fractions of entire composite consisting of each shell unit
% Outputs
%	 Condeff:    Scalar Effective conductivity 
%    R           vector of field concentration constants ordered from
%                outside-inward and first sphere/shale unit to second
%    f           vector of phase volume fractions ordered from
%                outside-inward and first sphere/shale unit to second
% ** note that f.*R=1
%
% Uses the function     FMINBND.m  to optimize equation:    SUM(fracs*(Condi - Condeff)*Ri=0;

% Written by Gary Mavko

global Condglobal fracsglobal 
fracsglobal = [f1,f2];

% normalize the fractions within each sphere/shell unit
fracs1 = fracs1/sum(fracs1);
fracs2 = fracs2/sum(fracs2);

% Find the exact effective conductivity of each sphere/shell unit
[Condeff1,Rmicro1] = shells_dielectric(fracs1,[],conds1);
[Condeff2,Rmicro2] = shells_dielectric(fracs2,[],conds2);
Condglobal = [Condeff1(1), Condeff2(1)];

R=[Rmicro1, Rmicro2];
f=[f1.*fracs1,f2*fracs2];

% Solve the Self-Consisten model for the overall effective conductivity due
% to mixing units 1 and 2, in the fractions f1 and f2

% in the FMINBND function: 
%    Condefftot is the effective conductivity of the entire composite: 
%    function deltacond contains the equation to minimize:  SUM(fi*(Condeffi - Condefftot)*Ri=0;
%    min(Cond) and max(Cond) define the range of solutions

[Condeff,Fval,exitflag] = fminbnd(@(Condefftot) deltacond(Condefftot),min([Condeff1,Condeff2]),max([Condeff1,Condeff2]));
if exitflag~=1, 'Cannot find a solution', end;

if f1 == 1,
    Condeff = Condeff1(1);
elseif f1 == 0,
    Condeff = Condeff2(1);
end;

    



% find distribution of electric fields
% first, find average field within each sphere/shell unit
Rmacro1 = (Condeff-Condeff2)/(f1*(Condeff1-Condeff2));
Rmacro2 = (1-f1*Rmacro1)/f2;

R1 = Rmacro1*Rmicro1;
R2 = Rmacro2*Rmicro2;



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function X=deltacond(C)
% 
global Condglobal  fracsglobal 
% 
% find the Field Concentration Factor R of each inclusion in the effective
% background conductivity.  THIS VERSION ONLY FOR SPHERES
for i = 1:length(fracsglobal)
    [R(i)] = Rfactor_cond_iso(C, Condglobal(i),1);
end
X = abs(sum(fracsglobal.*(Condglobal - C).*R));






