function [condeff, por]=cond_iso_inclusion_dem(cond1,cond2,asp,fracs,phic)
% [condeff, por]=cond_iso_inclusion_dem(cond1,cond2,asp,fracs,phic)
% Effective ELECTRICAL CONDUCTIVITY for ISOTROPIC composite of spheroid
% inclusions using the Differential Effective Medium method (DEM) .
% This version allows multiple spheroid inclusions, but they must all have the same
% fill material.
%
%   cond1      Conductivity of host material
%	cond2      Conductivity of the inclusion material. All inclusions must be the same
%	asp        Vector of Aspect ratios for the N inclusion types.
%              <1 for oblate spheroids
%   fracs      Vector of volume proportion of each inclusion type. Sum(frac) should be 1.
%   phic       Critical porosity, where HS bound will be computed and used
%              as inclusion material
%	condeff    Scalar Effective conductivity 
%
% Multiple inclusions are handled in the DEM algorithm by adding a portion
% of each type at each increment of porosity.  The vector fracs determines
% the volume fraction of each inclusion type, so that at porosity phi,
% the porosity of each inclusion type is phi*fracs.  Similarly, if an
% increment of porosity dphi is added at each time step, then dphi*fracs is
% the incremet of each inclusion type added.

% Written by Gary Mavko

% Check validity of inputs
if length(fracs) ~= length(asp) ,
    'input vectors asp and frac must have the same lengths'
    return;
end;
if length(cond2) ~= 1 ,
    'input cond2 must be a scalar'
    return;
end;
if phic <0 | phic > 1, 
    'critical porosity must be between 0 and 1'
    return;
end;
if cond1<0 | cond2<0,
    'conductivities must be positive'
    return;
end;

global DEMINPTC;

DEMINPTC.cond1 = cond1;   % background material conductivity
DEMINPTC.cond2 = cond2;   % inclusion material conductivity
DEMINPTC.asp   = asp  ;   % aspect ratios
DEMINPTC.fracs = fracs;   % volume portions of each inclusion type
DEMINPTC.phic  = phic;    % critical porosity

% Compute the HS  bound at the critical porosity.  This becomes the
% inclusion material
cond2 = 1/(phic/(cond2+2*cond2) + (1-phic)/(cond1+2*cond2)) - 2*cond2;

% integrate the ODE of conductivity vs. porosity
[phiout, condeff]=ode45m('demyprimeN_cond_iso',0.00,0.99999,cond1,max(cond1,cond2)*1e-15);
por=phic*phiout;
clear DEMINPTC;

% plot result if there are no outputs specified.
if nargout==0
   figure; plot(por,condeff,'-g','linewidth', 1);
end;

