function [eeff,R1,R2,variance1,variance2]=cond_iso_inclusion_CM(e,asp,fracs)
% cond_iso_inclusion_CM - Effective dielectric constant using multi-component
% Clausius-Mosotti equation, which is an inclusion model
%
% Effective DIELECTRIC CONSTANT for ISOTROPIC composite of spheroid
% inclusions using the Clausius-Mosotti method (CM) .
% ** limited to small concentrations **
% This version allows multiple spheroid constituents.
% *** If there are only two materials, e.g mineral and fluid, then estimates are also made 
% of mean field and variance in the both phases.   If the
% inclusions have multiple aspect ratios, but the same pore fill, then
% estimates can still be made as an average over all of them.
% Inputs:
%	e:         Vector of the Dielectric constants of the N constituent phases (vector of length N)
%              Each phase is isotropic and has a scalar dielectric constant.
%              *** First phase is the background medium ***
%	asp:       Vector of aspect ratios for each of the N constituents
%			   < 1 for oblate spheroids; 
%              *** First element, aspect ratio of background, is ignored ***
%   fracs:     Volume fraction of each phase. Sum(fracs)=1
% Outputs
%	eeff:      Scalar Effective Dielectric Constant 
%   R1         field concentraton factor in the background with phase 1,  
%   R2         field concentration in phse 2, averaged over all inclusions
%              assumes that all inclusions have same dielectric constant
%   variance1,variance2   variances of the field in phases 1 and 2,
%                         assuming all inclusions have the same dielectric constant
%
% Written by Gary Mavko
%

if sum(fracs)~=1,
    'sum of fractions not equal to 1; Normalizing'
    fracs = fracs/sum(fracs);
end;
f1 = fracs(1);                % fraction of background
f2 = sum(fracs(2:end));       % fraction of all inclusions together

% Replace zero dielectric with very small dielectric (only matters for
% conductivity calculation, since dielectric constant >= 1)
e(e == 0)=1e-6;

% compute the effective dielectric constant
% first, find the  R factors for each phase  (in this notation, R is not
% exactly the field concentration
for k=1:length(e)
    R(k) = findR(e(1),e(k),asp(k));
end;
% equation appears to correct for an error in Berryman (factor of 3e in
% sum);  We use instead expression from Torquato
eeff = sum(fracs.*e.*R)/sum(fracs.*R);

%%%%%%%%%%%%
% Next, perturb each phase to estimate the derivatives for mean and variance
de = .01;
% first, increment the phases by de and find the R's
for k=1:length(e)
    Rde1(k) = findR(e(1)+de, e(k),     asp(k));
    Rde2(k) = findR(e(1),    e(k)+de,  asp(k));
end;
% next, compute the incremented effective dielectrics
eeffde1 = sum(fracs.*[e(1)+de,e(2:end)].*Rde1)/sum(fracs.*Rde1);
eeffde2 = sum(fracs.*[e(1),e(2:end)+de].*Rde2)/sum(fracs.*Rde2);
deeffde1 = (eeffde1-eeff)/de;
deeffde2 = (eeffde2-eeff)/de;

% find the mean fields
s1 = e(1);
s2 = e(2);
R1 = (eeff-s2)/(f1*(s1-s2));
R2 = (1-f1*R1)/f2;


% find the variances of the field in each of the phases
variance1 = deeffde1/f1-R1^2;
variance2 = deeffde2/f2-R2^2;


% Check to make sure that all of the inclusions have the same phase.  If
% not, void the means and variances

e(2) == e(2:end)
if prod(e(2) == e(2:end)) == 0,
    R1 = nan;
    R2 = nan;
    variance1 = nan;
    variance2 = nan;
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function R=findR(eback,einc,asp)
% field concentration factor from Berryman mixing laws

gamma = 1/asp;

if gamma > 1,     % oblate spheroid
    A33 = (gamma^2/(gamma^2-1))*(1 - (atan(sqrt(gamma^2-1)))/sqrt(gamma^2-1));
elseif gamma < 1, % prolate spheroid
    A33 = (gamma^2/(gamma^2-1))*(1 - 0.5*log((1+sqrt(1-gamma^2))/(1-sqrt(1-gamma^2)))/sqrt(1-gamma^2));
elseif gamma == 1,
    A33 = 1/3;
end;
A11 = (1-A33)/2;

R = (1/9)*(2/(A11*einc+(1-A11)*eback) + 1/(A33*einc+(1-A33)*eback));


