function [condE]=cond_aligned_inclusion_scm(cond,phi, asp)
% % VTI Electrical conductvity (Shafiro and Kachanov, 2000; Torquato, 2002;
% Apresyan and Vlosov, 2014)
% Calculates macroscopic effective electrical conductivity of VTI anisotropic composites 
% with VTI spheroidal inclusions sitting in a VTI background, using SELF CONSISTENT.
% Axis of symmetry of background conductivity,
% inclusion conductivities, and spheroid shapes are all aligned with the X3 axis. 
% Can have multiple inclusions, each with its own aspect ratio and inclusion fill material.
% All constituents are considered as inclusions.
%
% Inputs: 
%    cond  = Array of conductivities of the N phases, in the form CondI(1:N,1:3).
%            cond(j,:) = [cond1j cond2j cond3j] - row of principal conductivies of jth phase
%    phi   = Vector of N phases volume fractions (porosities);  sum(phi) = 1
%    asp   = Vector of N spheroid aspect ratios.  asp<1 is oblate
% Output: 
%    condE = Effective principal conductivities in the form [CondE1 CondE2 CondE3]

% Written by Gary Mavko

niter = 20;
nphases = length(asp);
tol = 1e-5;

% validate inputs
if size(cond,2) ~= 3, 
    'Conductivities must be in an array cond(1:N,1:3)'
    return; 
end;
if size(cond,1) ~= length(phi),
    'must have one set of conductivies for each phase fraction'
    return;
end;
if length(phi) ~= length(asp),
    'must have one aspect ratio for each phase fraction'
    return;
end;
if sum(phi) ~=1,
    'phase fractions must sum to 1; normalizing'
    phi = phi/sum(phi);
end;

% don't let conductivities be too small
cond(cond < 1e-9) = 1e-9;

% first guess is Voigt-Reuss-Hill average of constituents
% % cVoigt = (phi*cond);
% % cReuss = 1./(phi*(1./cond));
% % cVRH   = (cVoigt + cReuss)/2
% % condE  = cVRH;
condE  = (phi*cond + 1./(phi*(1./cond)))/2;

% Iterate
% Loop over each inclusion type to get field concentration tensors
for i = 1:niter
    dc = zeros(1,3);
    for j = 1:nphases,
        R = Rfactor_aniso(condE, cond(j,:),asp(j));
        dc = dc + phi(j)*(cond(j,:)-condE).*[R(1,1), R(2,2), R(3,3)];
    end;
    condE = condE + dc;
    if mean(abs(dc./condE)) < tol, break; end;
end;


