function [k,rho,vp]=co2prop(ti,pi)
%[K,RHO,VP]=CO2PROP(T,P)
%
%CO2 properties as a function of temperature and pressure. Interpolation 
%based on Z. Wang's compilation and measurements.
%Inputs: T, P: temperature (deg. Celcius), pressure (MPa)
%Outputs: K, RHO, VP: Bulk modulus (GPa), density (gm/cc), and ultrasonic
%P-wave velocity (m/s) of CO2 at T and P conditions.
%With no input arguments, plots the data, and brings up user interface.
%The CO2 data is in 'co2propdata.mat'

%Written by T. Mukerji 11/99

if nargin==2
load co2propdata.mat; pmpa=0.0069*p;
k=interp2(t,pmpa,co2bulk,ti,pi);
rho=interp2(t,pmpa,co2rho,ti,pi);
vp=interp2(t,pmpa,co2vel,ti,pi);
end;

if nargin==0, co2propinit
elseif (nargin==1 & ti== -1) co2action
end;


%---------------
function co2propinit

load co2propdata.mat; pmpa=0.0069*p;
t0=80; p0=10;
[k0,rho0,vp0]=co2prop(t0,p0);
hfig=figure;
set(hfig,'units','normalized','position',[0.20 0.25 0.75 0.75]);
set(hfig,'doublebuffer','on');
subplot(3,1,1)
h=mesh(t,pmpa,co2bulk);axis tight, box on, colorbar, grid off
set(h,'facecolor','none')
hold on; ud.hkpt=plot3(t0,p0,k0,'*r','markersize',10);
title('CO_2 bulk modulus');
xlabel('T (^oC)'); ylabel('P (MPa)'); zlabel('Bulk modulus (GPa)');
subplot(3,1,2)
h=mesh(t,pmpa,co2rho);axis tight, box on, colorbar, grid off
set(h,'facecolor','none')
hold on; ud.hrpt=plot3(t0,p0,rho0,'*r','markersize',10);
title('CO_2 density');
xlabel('T (^oC)'); ylabel('P (MPa)'); zlabel('density (gm/cc)');
subplot(3,1,3)
h=mesh(t,pmpa,co2vel);axis tight, box on, colorbar, grid off
set(h,'facecolor','none')
hold on; ud.hvpt=plot3(t0,p0,vp0,'*r','markersize',10);
title('CO_2 velocity');
xlabel('T (^oC)'); ylabel('P (MPa)'); zlabel('velocity (m/s)');


ud.htslider=uicontrol('style', 'slider',...
                   'units', 'normalized',...
                   'position',[0.01 0.005 0.15 0.025],...
                   'min',min(t),...
                   'max',max(t),...
                   'value',t0,...
                   'tag','tslider',...
                   'callback', 'co2prop(-1)'...
                   );
       uicontrol('style','text',...
                 'units', 'normalized',...
                 'position',[0.01 0.03 0.05 0.03],...
                 'string', 'T(C)'...
                 );

ud.httext=uicontrol('style','edit',...
                  'backgroundcolor',[1 1 1],...
                 'units', 'normalized',...
                 'position',[0.075 0.03 0.07 0.03],...
                 'tag','ttext',...
                 'string', num2str(t0),...
                 'callback', 'co2prop(-1)'...
                 );

ud.hpslider=uicontrol('style', 'slider',...
                   'units', 'normalized',...
                   'position',[0.17 0.005 0.15 0.025],...
                   'min',min(pmpa),...
                   'max',max(pmpa),...
                   'value',p0,...
                   'tag','pslider',...
                   'callback', 'co2prop(-1)'...
                   );
       uicontrol('style','text',...
                 'units', 'normalized',...
                 'position',[0.17 0.03 0.05 0.03],...
                 'string', 'P(MPa)'...
                 );

ud.hptext=uicontrol('style','edit',...
                  'backgroundcolor',[1 1 1],...
                 'units', 'normalized',...
                 'position',[0.25 0.03 0.07 0.03],...
                 'tag','ptext',...
                 'string', num2str(p0),...
                 'callback', 'co2prop(-1)'...
                 );

ud.hktext=uicontrol('style','text',...
                    'units','normalized',...
                    'position',[0.5 0.005 0.12 0.03],...
                    'string',['K= ',num2str(k0,2),' GPa']...
                    );
ud.hrtext=uicontrol('style','text',...
                    'units','normalized',...
                    'position',[0.63 0.005 0.12 0.03],...
                    'string',['den= ',num2str(rho0,2),' gm/cc']...
                    );
ud.hvtext=uicontrol('style','text',...
                    'units','normalized',...
                    'position',[0.76 0.005 0.12 0.03],...
                    'string',['Vp= ',num2str(vp0,3),' m/s']...
                    );



set(hfig,'userdata',ud);
%----------------------
function co2action

ud=get(gcf,'userdata');
hcbo=gcbo;
if hcbo==ud.htslider
set(ud.httext,'string',num2str(get(hcbo,'value'),4)); end;
if hcbo==ud.httext
set(ud.htslider,'value',str2num(get(hcbo,'string'))); end;
if hcbo==ud.hpslider
set(ud.hptext,'string',num2str(get(hcbo,'value'),4)); end;
if hcbo==ud.hptext
set(ud.hpslider,'value',str2num(get(hcbo,'string'))); end;

ti=get(ud.htslider,'value');
pi=get(ud.hpslider,'value');
[k,rho,vp]=co2prop(ti,pi);
set(ud.hkpt,'xdata',ti,'ydata',pi,'zdata',k);
set(ud.hrpt,'xdata',ti,'ydata',pi,'zdata',rho);
set(ud.hvpt,'xdata',ti,'ydata',pi,'zdata',vp);

set(ud.hktext, 'string',['K= ',num2str(k,2),' GPa']);
set(ud.hrtext, 'string',['den= ',num2str(rho,2),' gm/cc']);
set(ud.hvtext, 'string',['Vp= ',num2str(vp,3),' m/s']);

